/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.server;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Ordering;
import com.google.inject.Inject;
import java.io.Closeable;
import java.io.IOException;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Consumer;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import org.apache.druid.common.guava.SettableSupplier;
import org.apache.druid.java.util.common.ISE;
import org.apache.druid.java.util.common.Intervals;
import org.apache.druid.java.util.common.io.Closer;
import org.apache.druid.java.util.emitter.EmittingLogger;
import org.apache.druid.query.TableDataSource;
import org.apache.druid.segment.PhysicalSegmentInspector;
import org.apache.druid.segment.ReferenceCountedSegmentProvider;
import org.apache.druid.segment.Segment;
import org.apache.druid.segment.SegmentLazyLoadFailCallback;
import org.apache.druid.segment.join.table.IndexedTable;
import org.apache.druid.segment.join.table.ReferenceCountedIndexedTableProvider;
import org.apache.druid.segment.loading.SegmentCacheManager;
import org.apache.druid.segment.loading.SegmentLoadingException;
import org.apache.druid.server.metrics.SegmentRowCountDistribution;
import org.apache.druid.timeline.DataSegment;
import org.apache.druid.timeline.SegmentId;
import org.apache.druid.timeline.VersionedIntervalTimeline;
import org.apache.druid.timeline.partition.PartitionChunk;
import org.apache.druid.timeline.partition.ShardSpec;
import org.apache.druid.utils.CollectionUtils;

public class SegmentManager {
    private static final EmittingLogger log = new EmittingLogger(SegmentManager.class);
    private final SegmentCacheManager cacheManager;
    private final ConcurrentHashMap<String, DataSourceState> dataSources = new ConcurrentHashMap();

    @Inject
    public SegmentManager(SegmentCacheManager cacheManager) {
        this.cacheManager = cacheManager;
    }

    @VisibleForTesting
    Map<String, DataSourceState> getDataSources() {
        return this.dataSources;
    }

    public Set<String> getDataSourceNames() {
        return this.dataSources.keySet();
    }

    public Map<String, Long> getDataSourceSizes() {
        return CollectionUtils.mapValues(this.dataSources, DataSourceState::getTotalSegmentSize);
    }

    public Map<String, Long> getAverageRowCountForDatasource() {
        return CollectionUtils.mapValues(this.dataSources, DataSourceState::getAverageRowCount);
    }

    public Map<String, SegmentRowCountDistribution> getRowCountDistribution() {
        return CollectionUtils.mapValues(this.dataSources, rec$ -> ((DataSourceState)rec$).getSegmentRowCountDistribution());
    }

    public Map<String, Long> getDataSourceCounts() {
        return CollectionUtils.mapValues(this.dataSources, DataSourceState::getNumSegments);
    }

    public Optional<VersionedIntervalTimeline<String, ReferenceCountedSegmentProvider>> getTimeline(TableDataSource dataSource) {
        return Optional.ofNullable(this.dataSources.get(dataSource.getName())).map(DataSourceState::getTimeline);
    }

    public Optional<Stream<ReferenceCountedIndexedTableProvider>> getIndexedTables(TableDataSource dataSource) {
        return this.getTimeline(dataSource).map(timeline -> {
            Stream<ReferenceCountedIndexedTableProvider> segments = timeline.lookup(Intervals.ETERNITY).stream().flatMap(x -> StreamSupport.stream(x.getObject().payloads().spliterator(), false));
            ConcurrentHashMap tables = Optional.ofNullable(this.dataSources.get(dataSource.getName())).map(DataSourceState::getTablesLookup).orElseThrow(() -> new ISE("dataSource[%s] does not have IndexedTables", new Object[]{dataSource.getName()}));
            return segments.map(segment -> (ReferenceCountedIndexedTableProvider)tables.get(segment.getBaseSegment().getId())).filter(Objects::nonNull);
        });
    }

    public boolean hasIndexedTables(String dataSourceName) {
        if (this.dataSources.containsKey(dataSourceName)) {
            return this.dataSources.get((Object)dataSourceName).tablesLookup.size() > 0;
        }
        return false;
    }

    public void loadSegmentOnBootstrap(DataSegment dataSegment, SegmentLazyLoadFailCallback loadFailed) throws SegmentLoadingException, IOException {
        ReferenceCountedSegmentProvider segment;
        try {
            segment = this.cacheManager.getBootstrapSegment(dataSegment, loadFailed);
            if (segment == null) {
                throw new SegmentLoadingException("No segment adapter found for bootstrap segment[%s] with loadSpec[%s].", new Object[]{dataSegment.getId(), dataSegment.getLoadSpec()});
            }
        }
        catch (SegmentLoadingException e) {
            this.cacheManager.cleanup(dataSegment);
            throw e;
        }
        this.loadSegmentInternal(dataSegment, segment, this.cacheManager::loadSegmentIntoPageCacheOnBootstrap);
    }

    public void loadSegment(DataSegment dataSegment) throws SegmentLoadingException, IOException {
        ReferenceCountedSegmentProvider segment;
        try {
            segment = this.cacheManager.getSegment(dataSegment);
            if (segment == null) {
                throw new SegmentLoadingException("No segment adapter found for segment[%s] with loadSpec[%s].", new Object[]{dataSegment.getId(), dataSegment.getLoadSpec()});
            }
        }
        catch (SegmentLoadingException e) {
            this.cacheManager.cleanup(dataSegment);
            throw e;
        }
        this.loadSegmentInternal(dataSegment, segment, this.cacheManager::loadSegmentIntoPageCache);
    }

    private void loadSegmentInternal(DataSegment dataSegment, ReferenceCountedSegmentProvider segment, Consumer<DataSegment> pageCacheLoadFunction) throws IOException {
        SettableSupplier resultSupplier = new SettableSupplier();
        this.dataSources.compute(dataSegment.getDataSource(), (k, v) -> {
            DataSourceState dataSourceState = v == null ? new DataSourceState() : v;
            VersionedIntervalTimeline<String, ReferenceCountedSegmentProvider> loadedIntervals = dataSourceState.getTimeline();
            PartitionChunk entry = loadedIntervals.findChunk(dataSegment.getInterval(), (Object)dataSegment.getVersion(), dataSegment.getShardSpec().getPartitionNum());
            if (entry != null) {
                log.warn("Told to load an adapter for segment[%s] that already exists", new Object[]{dataSegment.getId()});
                resultSupplier.set((Object)false);
            } else {
                Segment baseSegment = segment.getBaseSegment();
                IndexedTable table = (IndexedTable)baseSegment.as(IndexedTable.class);
                if (table != null) {
                    if (dataSourceState.isEmpty() || dataSourceState.numSegments == (long)dataSourceState.tablesLookup.size()) {
                        dataSourceState.tablesLookup.put(baseSegment.getId(), new ReferenceCountedIndexedTableProvider(table));
                    } else {
                        log.error("Cannot load segment[%s] with IndexedTable, no existing segments are joinable", new Object[]{baseSegment.getId()});
                    }
                } else if (dataSourceState.tablesLookup.size() > 0) {
                    log.error("Cannot load segment[%s] without IndexedTable, all existing segments are joinable", new Object[]{baseSegment.getId()});
                }
                loadedIntervals.add(dataSegment.getInterval(), (Object)dataSegment.getVersion(), dataSegment.getShardSpec().createChunk((Object)segment));
                PhysicalSegmentInspector countInspector = (PhysicalSegmentInspector)baseSegment.as(PhysicalSegmentInspector.class);
                long numOfRows = dataSegment.isTombstone() || countInspector == null ? 0L : (long)countInspector.getNumRows();
                dataSourceState.addSegment(dataSegment, numOfRows);
                pageCacheLoadFunction.accept(dataSegment);
                resultSupplier.set((Object)true);
            }
            return dataSourceState;
        });
        boolean loadResult = (Boolean)resultSupplier.get();
        if (loadResult) {
            this.cacheManager.storeInfoFile(dataSegment);
        }
    }

    public void dropSegment(DataSegment segment) {
        String dataSource = segment.getDataSource();
        this.dataSources.compute(dataSource, (dataSourceName, dataSourceState) -> {
            block12: {
                ReferenceCountedSegmentProvider oldSegmentRef;
                if (dataSourceState == null) {
                    log.info("Told to delete a queryable for a dataSource[%s] that doesn't exist.", new Object[]{dataSourceName});
                    return null;
                }
                VersionedIntervalTimeline<String, ReferenceCountedSegmentProvider> loadedIntervals = dataSourceState.getTimeline();
                ShardSpec shardSpec = segment.getShardSpec();
                PartitionChunk removed = loadedIntervals.remove(segment.getInterval(), (Object)segment.getVersion(), segment.getShardSpec().createChunk((Object)ReferenceCountedSegmentProvider.wrapSegment(null, (ShardSpec)shardSpec)));
                ReferenceCountedSegmentProvider referenceCountedSegmentProvider = oldSegmentRef = removed == null ? null : (ReferenceCountedSegmentProvider)removed.getObject();
                if (oldSegmentRef != null) {
                    try (Closer closer = Closer.create();){
                        PhysicalSegmentInspector countInspector = (PhysicalSegmentInspector)oldSegmentRef.getBaseSegment().as(PhysicalSegmentInspector.class);
                        long numOfRows = segment.isTombstone() || countInspector == null ? 0L : (long)countInspector.getNumRows();
                        dataSourceState.removeSegment(segment, numOfRows);
                        closer.register((Closeable)oldSegmentRef);
                        log.info("Attempting to close segment[%s]", new Object[]{segment.getId()});
                        ReferenceCountedIndexedTableProvider oldTable = dataSourceState.tablesLookup.remove(segment.getId());
                        if (oldTable != null) {
                            closer.register((Closeable)oldTable);
                        }
                        break block12;
                    }
                    catch (IOException e) {
                        throw new RuntimeException(e);
                    }
                }
                log.info("Told to delete a queryable on dataSource[%s] for interval[%s] and version[%s] that I don't have.", new Object[]{dataSourceName, segment.getInterval(), segment.getVersion()});
            }
            if (dataSourceState.isEmpty()) {
                return null;
            }
            DataSourceState dataSourceState2 = dataSourceState;
            return dataSourceState2;
        });
        this.cacheManager.removeInfoFile(segment);
        this.cacheManager.cleanup(segment);
    }

    public boolean canHandleSegments() {
        return this.cacheManager.canHandleSegments();
    }

    public List<DataSegment> getCachedSegments() throws IOException {
        return this.cacheManager.getCachedSegments();
    }

    public void shutdownBootstrap() {
        this.cacheManager.shutdownBootstrap();
    }

    public static class DataSourceState {
        private final VersionedIntervalTimeline<String, ReferenceCountedSegmentProvider> timeline = new VersionedIntervalTimeline((Comparator)Ordering.natural());
        private final ConcurrentHashMap<SegmentId, ReferenceCountedIndexedTableProvider> tablesLookup = new ConcurrentHashMap();
        private long totalSegmentSize;
        private long numSegments;
        private long rowCount;
        private final SegmentRowCountDistribution segmentRowCountDistribution = new SegmentRowCountDistribution();

        private void addSegment(DataSegment segment, long numOfRows) {
            this.totalSegmentSize += segment.getSize();
            ++this.numSegments;
            this.rowCount += numOfRows;
            if (segment.isTombstone()) {
                this.segmentRowCountDistribution.addTombstoneToDistribution();
            } else {
                this.segmentRowCountDistribution.addRowCountToDistribution(numOfRows);
            }
        }

        private void removeSegment(DataSegment segment, long numOfRows) {
            this.totalSegmentSize -= segment.getSize();
            --this.numSegments;
            this.rowCount -= numOfRows;
            if (segment.isTombstone()) {
                this.segmentRowCountDistribution.removeTombstoneFromDistribution();
            } else {
                this.segmentRowCountDistribution.removeRowCountFromDistribution(numOfRows);
            }
        }

        public VersionedIntervalTimeline<String, ReferenceCountedSegmentProvider> getTimeline() {
            return this.timeline;
        }

        public ConcurrentHashMap<SegmentId, ReferenceCountedIndexedTableProvider> getTablesLookup() {
            return this.tablesLookup;
        }

        public long getAverageRowCount() {
            return this.numSegments == 0L ? 0L : this.rowCount / this.numSegments;
        }

        public long getTotalSegmentSize() {
            return this.totalSegmentSize;
        }

        public long getNumSegments() {
            return this.numSegments;
        }

        public boolean isEmpty() {
            return this.numSegments == 0L;
        }

        private SegmentRowCountDistribution getSegmentRowCountDistribution() {
            return this.segmentRowCountDistribution;
        }
    }
}

