/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hive.kafka;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import java.io.IOException;
import java.time.Duration;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hive.kafka.HiveKafkaProducer;
import org.apache.hadoop.hive.kafka.KafkaUtils;
import org.apache.hadoop.hive.kafka.KafkaWritable;
import org.apache.hadoop.hive.ql.exec.FileSinkOperator;
import org.apache.hadoop.io.BytesWritable;
import org.apache.hadoop.io.Writable;
import org.apache.hadoop.mapred.RecordWriter;
import org.apache.hadoop.mapred.Reporter;
import org.apache.kafkaesque.clients.producer.Callback;
import org.apache.kafkaesque.common.KafkaException;
import org.apache.kafkaesque.common.TopicPartition;
import org.apache.kafkaesque.common.errors.AuthenticationException;
import org.apache.kafkaesque.common.errors.OutOfOrderSequenceException;
import org.apache.kafkaesque.common.errors.ProducerFencedException;
import org.apache.kafkaesque.common.errors.TimeoutException;
import org.apache.kafkaesque.common.serialization.ByteArraySerializer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class TransactionalKafkaWriter
implements FileSinkOperator.RecordWriter,
RecordWriter<BytesWritable, KafkaWritable> {
    private static final Logger LOG = LoggerFactory.getLogger(TransactionalKafkaWriter.class);
    private static final String TRANSACTION_DIR = "transaction_states";
    private static final Duration DURATION_0 = Duration.ofMillis(0L);
    private final String topic;
    private final HiveKafkaProducer<byte[], byte[]> producer;
    private final Callback callback;
    private final AtomicReference<Exception> sendExceptionRef = new AtomicReference();
    private final Path openTxFileName;
    private final boolean optimisticCommit;
    private final FileSystem fileSystem;
    private final Map<TopicPartition, Long> offsets = new HashMap<TopicPartition, Long>();
    private final String writerIdTopicId;
    private final long producerId;
    private final short producerEpoch;
    private long sentRecords = 0L;

    TransactionalKafkaWriter(String topic, Properties producerProperties, Path queryWorkingPath, FileSystem fileSystem, @Nullable Boolean optimisticCommit) {
        this.fileSystem = fileSystem;
        this.topic = (String)Preconditions.checkNotNull((Object)topic, (Object)"NULL topic !!");
        Preconditions.checkState((producerProperties.getProperty("bootstrap.servers") != null ? 1 : 0) != 0, (Object)"set [bootstrap.servers] property");
        producerProperties.setProperty("value.serializer", ByteArraySerializer.class.getName());
        producerProperties.setProperty("key.serializer", ByteArraySerializer.class.getName());
        this.producer = new HiveKafkaProducer(producerProperties);
        this.optimisticCommit = optimisticCommit == null ? true : optimisticCommit;
        this.callback = (metadata, exception) -> {
            if (exception != null) {
                this.sendExceptionRef.compareAndSet(null, exception);
            } else {
                TopicPartition tp = new TopicPartition(metadata.topic(), metadata.partition());
                this.offsets.put(tp, metadata.offset());
            }
        };
        assert (this.producer.getTransactionalId() != null);
        try {
            this.producer.initTransactions();
            this.producer.beginTransaction();
        }
        catch (Exception exception2) {
            this.logHints(exception2);
            if (this.tryToAbortTx(exception2)) {
                LOG.error("Aborting Transaction [{}] cause by ERROR [{}]", (Object)this.producer.getTransactionalId(), (Object)exception2.getMessage());
                this.producer.abortTransaction();
            }
            LOG.error("Closing writer [{}] caused by ERROR [{}]", (Object)this.producer.getTransactionalId(), (Object)exception2.getMessage());
            this.producer.close(DURATION_0);
            throw exception2;
        }
        this.writerIdTopicId = String.format("WriterId [%s], Kafka Topic [%s]", this.producer.getTransactionalId(), topic);
        this.producerEpoch = (short)(this.optimisticCommit ? -1 : (int)this.producer.getEpoch());
        this.producerId = this.optimisticCommit ? -1L : this.producer.getProducerId();
        LOG.info("DONE with Initialization of {}, Epoch[{}], internal ID[{}]", new Object[]{this.writerIdTopicId, this.producerEpoch, this.producerId});
        this.openTxFileName = this.optimisticCommit ? null : new Path(new Path(new Path(queryWorkingPath, TRANSACTION_DIR), this.producer.getTransactionalId()), String.valueOf(this.producerEpoch));
    }

    public void write(Writable w) throws IOException {
        this.checkExceptions();
        try {
            ++this.sentRecords;
            this.producer.send(KafkaUtils.toProducerRecord(this.topic, (KafkaWritable)w), this.callback);
        }
        catch (Exception e) {
            if (this.tryToAbortTx(e)) {
                this.producer.abortTransaction();
            }
            this.producer.close(DURATION_0);
            this.sendExceptionRef.compareAndSet(null, e);
            this.checkExceptions();
        }
    }

    private void logHints(Exception e) {
        if (e instanceof TimeoutException) {
            LOG.error("Maybe Try to increase [`retry.backoff.ms`] to avoid this error [{}].", (Object)e.getMessage());
        }
    }

    public void close(boolean abort) throws IOException {
        if (abort) {
            LOG.warn("Aborting Transaction and Sending from {}", (Object)this.writerIdTopicId);
            try {
                this.producer.abortTransaction();
            }
            catch (Exception e) {
                LOG.error("Aborting Transaction {} failed due to [{}]", (Object)this.writerIdTopicId, (Object)e.getMessage());
            }
            this.producer.close(DURATION_0);
            return;
        }
        LOG.info("Flushing Kafka buffer of writerId {}", (Object)this.writerIdTopicId);
        this.producer.flush();
        String formattedMsg = "Topic[%s] Partition [%s] -> Last offset [%s]";
        String flushedOffsetMsg = this.offsets.entrySet().stream().map(topicPartitionLongEntry -> String.format(formattedMsg, ((TopicPartition)topicPartitionLongEntry.getKey()).topic(), ((TopicPartition)topicPartitionLongEntry.getKey()).partition(), topicPartitionLongEntry.getValue())).collect(Collectors.joining(","));
        LOG.info("WriterId {} flushed the following [{}] ", (Object)this.writerIdTopicId, (Object)flushedOffsetMsg);
        this.checkExceptions();
        if (this.optimisticCommit) {
            this.commitTransaction();
        } else {
            this.persistTxState();
        }
        this.checkExceptions();
        LOG.info("Closed writerId [{}], Sent [{}] records to Topic [{}]", new Object[]{this.producer.getTransactionalId(), this.sentRecords, this.topic});
        this.producer.close(Duration.ZERO);
    }

    private void commitTransaction() {
        LOG.info("Attempting Optimistic commit by {}", (Object)this.writerIdTopicId);
        try {
            this.producer.commitTransaction();
        }
        catch (Exception e) {
            this.sendExceptionRef.compareAndSet(null, e);
        }
    }

    private void persistTxState() {
        LOG.info("Committing state to path [{}] by [{}]", (Object)this.openTxFileName.toString(), (Object)this.writerIdTopicId);
        try (FSDataOutputStream outStream = this.fileSystem.create(this.openTxFileName);){
            outStream.writeLong(this.producerId);
            outStream.writeShort((int)this.producerEpoch);
        }
        catch (Exception e) {
            this.sendExceptionRef.compareAndSet(null, e);
        }
    }

    public void write(BytesWritable bytesWritable, KafkaWritable kafkaWritable) throws IOException {
        this.write(kafkaWritable);
    }

    public void close(Reporter reporter) throws IOException {
        this.close(false);
    }

    @VisibleForTesting
    long getSentRecords() {
        return this.sentRecords;
    }

    @VisibleForTesting
    short getProducerEpoch() {
        return this.producerEpoch;
    }

    @VisibleForTesting
    long getProducerId() {
        return this.producerId;
    }

    private void checkExceptions() throws IOException {
        if (this.sendExceptionRef.get() != null && this.sendExceptionRef.get() instanceof KafkaException && this.sendExceptionRef.get().getCause() instanceof ProducerFencedException) {
            this.sendExceptionRef.updateAndGet(e -> (KafkaException)e.getCause());
        }
        if (this.sendExceptionRef.get() != null) {
            Exception exception = this.sendExceptionRef.get();
            this.logHints(exception);
            if (this.tryToAbortTx(exception)) {
                LOG.error("Aborting Transaction [{}] cause by ERROR [{}]", (Object)this.writerIdTopicId, (Object)exception.getMessage());
                this.producer.abortTransaction();
            }
            LOG.error("Closing writer [{}] caused by ERROR [{}]", (Object)this.writerIdTopicId, (Object)exception.getMessage());
            this.producer.close(DURATION_0);
            throw new IOException(exception);
        }
    }

    private boolean tryToAbortTx(Throwable e) {
        boolean isNotFencedOut = !(e instanceof ProducerFencedException) && !(e instanceof OutOfOrderSequenceException) && !(e instanceof AuthenticationException);
        boolean causeIsNotFencedOut = e.getCause() == null || !(e.getCause() instanceof ProducerFencedException);
        return isNotFencedOut && causeIsNotFencedOut;
    }

    static Map<String, Pair<Long, Short>> getTransactionsState(FileSystem fs, Path queryWorkingDir) throws IOException {
        Path transactionWorkingDir = new Path(queryWorkingDir, TRANSACTION_DIR);
        FileStatus[] files = fs.listStatus(transactionWorkingDir);
        Set transactionSet = Arrays.stream(files).filter(FileStatus::isDirectory).collect(Collectors.toSet());
        Set<Path> setOfTxPath = transactionSet.stream().map(FileStatus::getPath).collect(Collectors.toSet());
        ImmutableMap.Builder builder = ImmutableMap.builder();
        setOfTxPath.forEach(path -> {
            String txId = path.getName();
            try {
                long internalId;
                FileStatus[] epochFiles = fs.listStatus(path);
                Optional<Short> maxEpoch = Arrays.stream(epochFiles).filter(FileStatus::isFile).map(fileStatus -> Short.valueOf(fileStatus.getPath().getName())).max(Short::compareTo);
                short epoch = maxEpoch.orElseThrow(() -> new RuntimeException("Missing sub directory epoch from directory [" + path.toString() + "]"));
                Path openTxFileName = new Path(path, String.valueOf(epoch));
                try (FSDataInputStream inStream = fs.open(openTxFileName);){
                    internalId = inStream.readLong();
                    short fileEpoch = inStream.readShort();
                    if (epoch != fileEpoch) {
                        throw new RuntimeException(String.format("Was expecting [%s] but got [%s] from path [%s]", epoch, fileEpoch, path.toString()));
                    }
                }
                catch (IOException e) {
                    throw new RuntimeException(e);
                }
                builder.put((Object)txId, (Object)Pair.of((Object)internalId, (Object)epoch));
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        });
        return builder.build();
    }
}

