/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.storageengine.dataregion.wal.buffer;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Predicate;
import org.apache.iotdb.commons.concurrent.IoTDBThreadPoolFactory;
import org.apache.iotdb.commons.concurrent.ThreadName;
import org.apache.iotdb.commons.conf.CommonDescriptor;
import org.apache.iotdb.db.conf.IoTDBConfig;
import org.apache.iotdb.db.conf.IoTDBDescriptor;
import org.apache.iotdb.db.queryengine.plan.planner.plan.node.write.DeleteDataNode;
import org.apache.iotdb.db.queryengine.plan.planner.plan.node.write.InsertNode;
import org.apache.iotdb.db.service.metrics.WritingMetrics;
import org.apache.iotdb.db.storageengine.dataregion.wal.buffer.AbstractWALBuffer;
import org.apache.iotdb.db.storageengine.dataregion.wal.buffer.IWALByteBufferView;
import org.apache.iotdb.db.storageengine.dataregion.wal.buffer.WALEntry;
import org.apache.iotdb.db.storageengine.dataregion.wal.buffer.WALEntryType;
import org.apache.iotdb.db.storageengine.dataregion.wal.buffer.WALSignalEntry;
import org.apache.iotdb.db.storageengine.dataregion.wal.checkpoint.Checkpoint;
import org.apache.iotdb.db.storageengine.dataregion.wal.checkpoint.CheckpointManager;
import org.apache.iotdb.db.storageengine.dataregion.wal.exception.BrokenWALFileException;
import org.apache.iotdb.db.storageengine.dataregion.wal.exception.WALNodeClosedException;
import org.apache.iotdb.db.storageengine.dataregion.wal.io.WALMetaData;
import org.apache.iotdb.db.storageengine.dataregion.wal.utils.MemoryControlledWALEntryQueue;
import org.apache.iotdb.db.storageengine.dataregion.wal.utils.WALFileStatus;
import org.apache.iotdb.db.storageengine.dataregion.wal.utils.WALFileUtils;
import org.apache.iotdb.db.storageengine.dataregion.wal.utils.WALMode;
import org.apache.iotdb.db.storageengine.dataregion.wal.utils.listener.WALFlushListener;
import org.apache.iotdb.db.utils.MmapUtil;
import org.apache.tsfile.compress.ICompressor;
import org.apache.tsfile.file.metadata.enums.CompressionType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class WALBuffer
extends AbstractWALBuffer {
    private static final Logger logger = LoggerFactory.getLogger(WALBuffer.class);
    private static final IoTDBConfig config = IoTDBDescriptor.getInstance().getConfig();
    public static final int ONE_THIRD_WAL_BUFFER_SIZE = config.getWalBufferSize() / 3;
    private static final double FSYNC_BUFFER_RATIO = 0.95;
    private static final WritingMetrics WRITING_METRICS = WritingMetrics.getInstance();
    private volatile boolean isClosed = false;
    private final CheckpointManager checkpointManager;
    private final MemoryControlledWALEntryQueue walEntries = new MemoryControlledWALEntryQueue();
    private final Lock buffersLock = new ReentrantLock();
    private final Condition idleBufferReadyCondition = this.buffersLock.newCondition();
    private long lastFsyncPosition;
    private volatile ByteBuffer workingBuffer;
    private volatile ByteBuffer idleBuffer;
    private volatile ByteBuffer syncingBuffer;
    private ByteBuffer compressedByteBuffer;
    protected volatile WALFileStatus currentFileStatus;
    private final ExecutorService serializeThread;
    private final ExecutorService syncBufferThread;
    private final Map<Long, Set<Long>> memTableIdsOfWal = new ConcurrentHashMap<Long, Set<Long>>();

    public WALBuffer(String identifier, String logDirectory) throws IOException {
        this(identifier, logDirectory, new CheckpointManager(identifier, logDirectory), 0L, 0L);
    }

    public WALBuffer(String identifier, String logDirectory, CheckpointManager checkpointManager, long startFileVersion, long startSearchIndex) throws IOException {
        super(identifier, logDirectory, startFileVersion, startSearchIndex);
        this.checkpointManager = checkpointManager;
        this.currentFileStatus = WALFileStatus.CONTAINS_NONE_SEARCH_INDEX;
        this.allocateBuffers();
        this.currentWALFileWriter.setCompressedByteBuffer(this.compressedByteBuffer);
        this.serializeThread = IoTDBThreadPoolFactory.newSingleThreadExecutor((String)(ThreadName.WAL_SERIALIZE.getName() + "(node-" + identifier + ")"));
        this.syncBufferThread = IoTDBThreadPoolFactory.newSingleThreadExecutor((String)(ThreadName.WAL_SYNC.getName() + "(node-" + identifier + ")"));
        this.serializeThread.submit(new SerializeTask());
    }

    private void allocateBuffers() {
        try {
            this.workingBuffer = ByteBuffer.allocateDirect(ONE_THIRD_WAL_BUFFER_SIZE);
            this.idleBuffer = ByteBuffer.allocateDirect(ONE_THIRD_WAL_BUFFER_SIZE);
            this.compressedByteBuffer = ByteBuffer.allocateDirect(this.getCompressedByteBufferSize(ONE_THIRD_WAL_BUFFER_SIZE));
        }
        catch (OutOfMemoryError e) {
            logger.error("Fail to allocate wal node-{}'s buffer because out of memory.", (Object)this.identifier, (Object)e);
            this.close();
            throw e;
        }
    }

    private int getCompressedByteBufferSize(int size) {
        return ICompressor.getCompressor((CompressionType)CompressionType.LZ4).getMaxBytesForCompression(size);
    }

    @Override
    protected File rollLogWriter(long searchIndex, WALFileStatus fileStatus) throws IOException {
        File file = super.rollLogWriter(searchIndex, fileStatus);
        this.currentWALFileWriter.setCompressedByteBuffer(this.compressedByteBuffer);
        return file;
    }

    public void setBufferSize(int size) {
        int capacity = size / 3;
        this.buffersLock.lock();
        try {
            MmapUtil.clean(this.workingBuffer);
            MmapUtil.clean(this.workingBuffer);
            MmapUtil.clean(this.syncingBuffer);
            MmapUtil.clean(this.compressedByteBuffer);
            this.workingBuffer = ByteBuffer.allocateDirect(capacity);
            this.idleBuffer = ByteBuffer.allocateDirect(capacity);
            this.compressedByteBuffer = ByteBuffer.allocateDirect(this.getCompressedByteBufferSize(capacity));
            this.currentWALFileWriter.setCompressedByteBuffer(this.compressedByteBuffer);
        }
        catch (OutOfMemoryError e) {
            logger.error("Fail to allocate wal node-{}'s buffer because out of memory.", (Object)this.identifier, (Object)e);
            this.close();
            throw e;
        }
        finally {
            this.buffersLock.unlock();
        }
    }

    @Override
    public void write(WALEntry walEntry) {
        if (this.isClosed) {
            logger.warn("Fail to write WALEntry into wal node-{} because this node is closed. It's ok to see this log during data region deletion.", (Object)this.identifier);
            walEntry.getWalFlushListener().fail((Exception)((Object)new WALNodeClosedException(this.identifier)));
            return;
        }
        try {
            this.walEntries.put(walEntry);
        }
        catch (InterruptedException e) {
            logger.warn("Interrupted when waiting for adding WALEntry to buffer.");
            Thread.currentThread().interrupt();
        }
    }

    private void syncWorkingBuffer(long searchIndex, WALFileStatus fileStatus) {
        this.switchWorkingBufferToFlushing();
        this.syncBufferThread.submit(new SyncBufferTask(searchIndex, fileStatus, false));
        this.currentFileStatus = WALFileStatus.CONTAINS_NONE_SEARCH_INDEX;
    }

    private void fsyncWorkingBuffer(long searchIndex, WALFileStatus fileStatus, SerializeInfo info) {
        this.switchWorkingBufferToFlushing();
        this.syncBufferThread.submit(new SyncBufferTask(searchIndex, fileStatus, true, info));
        this.currentFileStatus = WALFileStatus.CONTAINS_NONE_SEARCH_INDEX;
    }

    private void switchWorkingBufferToFlushing() {
        this.buffersLock.lock();
        try {
            while (this.idleBuffer == null) {
                this.idleBufferReadyCondition.await();
            }
            this.syncingBuffer = this.workingBuffer;
            this.workingBuffer = this.idleBuffer;
            this.workingBuffer.clear();
            this.idleBuffer = null;
        }
        catch (InterruptedException e) {
            logger.warn("Interrupted When waiting for available working buffer.");
            Thread.currentThread().interrupt();
        }
        finally {
            this.buffersLock.unlock();
        }
    }

    private void switchSyncingBufferToIdle() {
        this.buffersLock.lock();
        try {
            this.idleBuffer = this.syncingBuffer;
            this.syncingBuffer = null;
            this.idleBufferReadyCondition.signalAll();
        }
        finally {
            this.buffersLock.unlock();
        }
    }

    @Override
    public void waitForFlush() throws InterruptedException {
        this.buffersLock.lock();
        try {
            this.idleBufferReadyCondition.await();
        }
        finally {
            this.buffersLock.unlock();
        }
    }

    @Override
    public void waitForFlush(Predicate<WALBuffer> waitPredicate) throws InterruptedException {
        this.buffersLock.lock();
        try {
            if (waitPredicate.test(this)) {
                this.idleBufferReadyCondition.await();
            }
        }
        finally {
            this.buffersLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public boolean waitForFlush(long time, TimeUnit unit) throws InterruptedException {
        this.buffersLock.lock();
        try {
            boolean bl = this.idleBufferReadyCondition.await(time, unit);
            return bl;
        }
        finally {
            this.buffersLock.unlock();
        }
    }

    @Override
    public void close() {
        if (this.serializeThread != null) {
            try {
                this.walEntries.put(new WALSignalEntry(WALEntryType.CLOSE_SIGNAL));
            }
            catch (InterruptedException e) {
                logger.error("Fail to put CLOSE_SIGNAL to walEntries.", (Throwable)e);
                Thread.currentThread().interrupt();
            }
            this.isClosed = true;
            this.shutdownThread(this.serializeThread, ThreadName.WAL_SERIALIZE);
        }
        if (this.syncBufferThread != null) {
            this.shutdownThread(this.syncBufferThread, ThreadName.WAL_SYNC);
        }
        if (this.currentWALFileWriter != null) {
            try {
                this.currentWALFileWriter.close();
            }
            catch (IOException e) {
                logger.error("Fail to close wal node-{}'s log writer.", (Object)this.identifier, (Object)e);
            }
        }
        this.checkpointManager.close();
        MmapUtil.clean(this.workingBuffer);
        MmapUtil.clean(this.workingBuffer);
        MmapUtil.clean(this.syncingBuffer);
        MmapUtil.clean(this.compressedByteBuffer);
    }

    private void shutdownThread(ExecutorService thread, ThreadName threadName) {
        thread.shutdown();
        try {
            if (!thread.awaitTermination(30L, TimeUnit.SECONDS)) {
                logger.warn("Waiting thread {} to be terminated is timeout", (Object)threadName.getName());
            }
        }
        catch (InterruptedException e) {
            logger.warn("Thread {} still doesn't exit after 30s", (Object)threadName.getName());
            Thread.currentThread().interrupt();
        }
    }

    @Override
    public boolean isAllWALEntriesConsumed() {
        this.buffersLock.lock();
        try {
            boolean bl = this.walEntries.isEmpty() && this.workingBuffer.position() == 0 && this.syncingBuffer == null;
            return bl;
        }
        finally {
            this.buffersLock.unlock();
        }
    }

    public CheckpointManager getCheckpointManager() {
        return this.checkpointManager;
    }

    public void removeMemTableIdsOfWal(Long walVersionId) {
        this.memTableIdsOfWal.remove(walVersionId);
    }

    public Set<Long> getMemTableIds(long fileVersionId) {
        if (fileVersionId >= this.currentWALFileVersion) {
            return null;
        }
        return this.memTableIdsOfWal.computeIfAbsent(fileVersionId, id -> {
            try {
                File file = WALFileUtils.getWALFile(new File(this.logDirectory), id);
                return WALMetaData.readFromWALFile(file, FileChannel.open(file.toPath(), StandardOpenOption.READ)).getMemTablesId();
            }
            catch (BrokenWALFileException e) {
                logger.warn("Fail to read memTable ids from the wal file {} of wal node {}: {}", new Object[]{id, this.identifier, e.getMessage()});
            }
            catch (IOException e) {
                logger.warn("Fail to read memTable ids from the wal file {} of wal node {}.", new Object[]{id, this.identifier, e});
            }
            return Collections.emptySet();
        });
    }

    public Map<Long, Set<Long>> getMemTableIdsOfWal() {
        return this.memTableIdsOfWal;
    }

    private static class SerializeInfo {
        final WALMetaData metaData = new WALMetaData();
        final Map<Long, Long> memTableId2WalDiskUsage = new HashMap<Long, Long>();
        final List<Checkpoint> checkpoints = new ArrayList<Checkpoint>();
        final List<WALFlushListener> fsyncListeners = new ArrayList<WALFlushListener>();
        WALFlushListener rollWALFileWriterListener = null;

        private SerializeInfo() {
        }
    }

    private class SerializeTask
    implements Runnable {
        private final ByteBufferView byteBufferView;
        private final SerializeInfo info;
        private int totalSize;

        private SerializeTask() {
            this.byteBufferView = new ByteBufferView();
            this.info = new SerializeInfo();
            this.totalSize = 0;
        }

        @Override
        public void run() {
            try {
                this.serialize();
            }
            finally {
                if (!WALBuffer.this.isClosed) {
                    WALBuffer.this.serializeThread.submit(new SerializeTask());
                }
            }
        }

        private void serialize() {
            boolean returnFlag;
            long start = System.nanoTime();
            try {
                WALEntry firstWALEntry = WALBuffer.this.walEntries.take();
                returnFlag = this.handleWALEntry(firstWALEntry);
                if (returnFlag) {
                    WRITING_METRICS.recordSerializeWALEntryTotalCost(System.nanoTime() - start);
                    return;
                }
            }
            catch (InterruptedException e) {
                logger.warn("Interrupted when waiting for taking WALEntry from blocking queue to serialize.");
                Thread.currentThread().interrupt();
            }
            while ((double)this.totalSize < (double)ONE_THIRD_WAL_BUFFER_SIZE * 0.95) {
                WALEntry walEntry = null;
                try {
                    walEntry = config.getWalMode().equals((Object)WALMode.ASYNC) ? WALBuffer.this.walEntries.poll(config.getWalAsyncModeFsyncDelayInMs(), TimeUnit.MILLISECONDS) : WALBuffer.this.walEntries.poll(config.getWalSyncModeFsyncDelayInMs(), TimeUnit.MILLISECONDS);
                }
                catch (InterruptedException e) {
                    logger.warn("Interrupted when waiting for taking WALEntry from blocking queue to serialize.");
                    Thread.currentThread().interrupt();
                }
                if (walEntry == null) break;
                returnFlag = this.handleWALEntry(walEntry);
                if (!returnFlag) continue;
                WRITING_METRICS.recordSerializeWALEntryTotalCost(System.nanoTime() - start);
                return;
            }
            WRITING_METRICS.recordSerializeWALEntryTotalCost(System.nanoTime() - start);
            if (this.totalSize > 0 || !this.info.checkpoints.isEmpty()) {
                WALBuffer.this.fsyncWorkingBuffer(WALBuffer.this.currentSearchIndex, WALBuffer.this.currentFileStatus, this.info);
            }
        }

        private boolean handleWALEntry(WALEntry walEntry) {
            if (walEntry.isSignal()) {
                return this.handleSignalEntry((WALSignalEntry)walEntry);
            }
            this.handleInfoEntry(walEntry);
            return false;
        }

        private void handleInfoEntry(WALEntry walEntry) {
            int size;
            if (walEntry.getType() == WALEntryType.MEMORY_TABLE_CHECKPOINT) {
                this.info.checkpoints.add((Checkpoint)walEntry.getValue());
                return;
            }
            int startPosition = this.byteBufferView.position();
            try {
                walEntry.serialize(this.byteBufferView);
                size = this.byteBufferView.position() - startPosition;
            }
            catch (Exception e) {
                logger.error("Fail to serialize WALEntry to wal node-{}'s buffer, discard it.", (Object)WALBuffer.this.identifier, (Object)e);
                walEntry.getWalFlushListener().fail(e);
                return;
            }
            long searchIndex = -1L;
            if (walEntry.getType().needSearch() && (searchIndex = walEntry.getType() == WALEntryType.DELETE_DATA_NODE ? ((DeleteDataNode)walEntry.getValue()).getSearchIndex() : ((InsertNode)((Object)walEntry.getValue())).getSearchIndex()) != -1L) {
                WALBuffer.this.currentSearchIndex = searchIndex;
                WALBuffer.this.currentFileStatus = WALFileStatus.CONTAINS_SEARCH_INDEX;
            }
            this.totalSize += size;
            this.info.metaData.add(size, searchIndex, walEntry.getMemTableId());
            this.info.memTableId2WalDiskUsage.compute(walEntry.getMemTableId(), (k, v) -> v == null ? (long)size : v + (long)size);
            walEntry.getWalFlushListener().getWalEntryHandler().setSize(size);
            this.info.fsyncListeners.add(walEntry.getWalFlushListener());
        }

        private boolean handleSignalEntry(WALSignalEntry walSignalEntry) {
            switch (walSignalEntry.getType()) {
                case ROLL_WAL_LOG_WRITER_SIGNAL: {
                    if (logger.isDebugEnabled()) {
                        logger.debug("Handle roll log writer signal for wal node-{}.", (Object)WALBuffer.this.identifier);
                    }
                    this.info.rollWALFileWriterListener = walSignalEntry.getWalFlushListener();
                    WALBuffer.this.fsyncWorkingBuffer(WALBuffer.this.currentSearchIndex, WALBuffer.this.currentFileStatus, this.info);
                    return true;
                }
                case CLOSE_SIGNAL: {
                    boolean dataExists;
                    if (logger.isDebugEnabled()) {
                        logger.debug("Handle close signal for wal node-{}, there are {} entries left.", (Object)WALBuffer.this.identifier, (Object)WALBuffer.this.walEntries.size());
                    }
                    boolean bl = dataExists = this.totalSize > 0;
                    if (dataExists) {
                        WALBuffer.this.fsyncWorkingBuffer(WALBuffer.this.currentSearchIndex, WALBuffer.this.currentFileStatus, this.info);
                    }
                    WALBuffer.this.isClosed = true;
                    return dataExists;
                }
            }
            return false;
        }
    }

    private class SyncBufferTask
    implements Runnable {
        private final long searchIndex;
        private final WALFileStatus fileStatus;
        private final boolean forceFlag;
        private final SerializeInfo info;

        public SyncBufferTask(long searchIndex, WALFileStatus fileStatus, boolean forceFlag) {
            this(searchIndex, fileStatus, forceFlag, null);
        }

        public SyncBufferTask(long searchIndex, WALFileStatus fileStatus, boolean forceFlag, SerializeInfo info) {
            this.searchIndex = searchIndex;
            this.fileStatus = fileStatus;
            this.forceFlag = forceFlag;
            this.info = info == null ? new SerializeInfo() : info;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            long startTime = System.nanoTime();
            this.makeMemTableCheckpoints();
            long walFileVersionId = WALBuffer.this.currentWALFileVersion;
            WALBuffer.this.currentWALFileWriter.updateFileStatus(this.fileStatus);
            double usedRatio = (double)WALBuffer.this.syncingBuffer.position() / (double)WALBuffer.this.syncingBuffer.capacity();
            WRITING_METRICS.recordWALBufferUsedRatio(usedRatio);
            logger.debug("Sync wal buffer, forceFlag: {}, buffer used: {} / {} = {}%", new Object[]{this.forceFlag, WALBuffer.this.syncingBuffer.position(), WALBuffer.this.syncingBuffer.capacity(), usedRatio * 100.0});
            double compressionRatio = 1.0;
            try {
                compressionRatio = WALBuffer.this.currentWALFileWriter.write(WALBuffer.this.syncingBuffer, this.info.metaData);
            }
            catch (Throwable e) {
                logger.error("Fail to sync wal node-{}'s buffer, change system mode to error.", (Object)WALBuffer.this.identifier, (Object)e);
                CommonDescriptor.getInstance().getConfig().handleUnrecoverableError();
            }
            finally {
                WALBuffer.this.switchSyncingBufferToIdle();
            }
            WALBuffer.this.memTableIdsOfWal.computeIfAbsent(WALBuffer.this.currentWALFileVersion, memTableIds -> new HashSet()).addAll(this.info.metaData.getMemTablesId());
            WALBuffer.this.checkpointManager.updateCostOfActiveMemTables(this.info.memTableId2WalDiskUsage, compressionRatio);
            boolean forceSuccess = false;
            if (this.info.rollWALFileWriterListener != null || this.forceFlag && WALBuffer.this.currentWALFileWriter.originalSize() >= config.getWalFileSizeThresholdInByte()) {
                try {
                    WALBuffer.this.rollLogWriter(this.searchIndex, WALBuffer.this.currentWALFileWriter.getWalFileStatus());
                    forceSuccess = true;
                    if (this.info.rollWALFileWriterListener != null) {
                        this.info.rollWALFileWriterListener.succeed();
                    }
                }
                catch (IOException e) {
                    logger.error("Fail to roll wal node-{}'s log writer, change system mode to error.", (Object)WALBuffer.this.identifier, (Object)e);
                    if (this.info.rollWALFileWriterListener != null) {
                        this.info.rollWALFileWriterListener.fail(e);
                    }
                    CommonDescriptor.getInstance().getConfig().handleUnrecoverableError();
                }
            } else if (this.forceFlag) {
                try {
                    WALBuffer.this.currentWALFileWriter.force();
                    forceSuccess = true;
                }
                catch (IOException e) {
                    logger.error("Fail to fsync wal node-{}'s log writer, change system mode to error.", (Object)WALBuffer.this.identifier, (Object)e);
                    for (WALFlushListener fsyncListener : this.info.fsyncListeners) {
                        fsyncListener.fail(e);
                    }
                    CommonDescriptor.getInstance().getConfig().handleUnrecoverableError();
                }
            }
            if (forceSuccess) {
                long position = WALBuffer.this.lastFsyncPosition;
                for (WALFlushListener fsyncListener : this.info.fsyncListeners) {
                    fsyncListener.succeed();
                    if (fsyncListener.getWalEntryHandler() == null) continue;
                    fsyncListener.getWalEntryHandler().setEntryPosition(walFileVersionId, position);
                    position += (long)fsyncListener.getWalEntryHandler().getSize();
                }
                WALBuffer.this.lastFsyncPosition = WALBuffer.this.currentWALFileWriter.originalSize();
            }
            WRITING_METRICS.recordWALBufferEntriesCount(this.info.fsyncListeners.size());
            WRITING_METRICS.recordSyncWALBufferCost(System.nanoTime() - startTime, this.forceFlag);
        }

        private void makeMemTableCheckpoints() {
            if (this.info.checkpoints.isEmpty()) {
                return;
            }
            block4: for (Checkpoint checkpoint : this.info.checkpoints) {
                switch (checkpoint.getType()) {
                    case CREATE_MEMORY_TABLE: {
                        WALBuffer.this.checkpointManager.makeCreateMemTableCPOnDisk(checkpoint.getMemTableInfos().get(0).getMemTableId());
                        continue block4;
                    }
                    case FLUSH_MEMORY_TABLE: {
                        WALBuffer.this.checkpointManager.makeFlushMemTableCP(checkpoint.getMemTableInfos().get(0).getMemTableId());
                        continue block4;
                    }
                }
                throw new RuntimeException("Cannot make other checkpoint types in the wal buffer, type is " + (Object)((Object)checkpoint.getType()));
            }
            WALBuffer.this.checkpointManager.fsyncCheckpointFile();
        }
    }

    private class ByteBufferView
    implements IWALByteBufferView {
        private int flushedBytesNum = 0;

        private ByteBufferView() {
        }

        private void ensureEnoughSpace(int bytesNum) {
            if (WALBuffer.this.workingBuffer.remaining() < bytesNum) {
                this.rollBuffer();
            }
        }

        private void rollBuffer() {
            this.flushedBytesNum += WALBuffer.this.workingBuffer.position();
            WALBuffer.this.syncWorkingBuffer(WALBuffer.this.currentSearchIndex, WALBuffer.this.currentFileStatus);
        }

        @Override
        public void put(byte b) {
            this.ensureEnoughSpace(1);
            WALBuffer.this.workingBuffer.put(b);
        }

        @Override
        public void put(byte[] src) {
            int needCapacity;
            int offset = 0;
            while (true) {
                int leftCapacity;
                if ((leftCapacity = WALBuffer.this.workingBuffer.remaining()) >= (needCapacity = src.length - offset)) break;
                WALBuffer.this.workingBuffer.put(src, offset, leftCapacity);
                offset += leftCapacity;
                this.rollBuffer();
            }
            WALBuffer.this.workingBuffer.put(src, offset, needCapacity);
        }

        @Override
        public void putChar(char value) {
            this.ensureEnoughSpace(2);
            WALBuffer.this.workingBuffer.putChar(value);
        }

        @Override
        public void putShort(short value) {
            this.ensureEnoughSpace(2);
            WALBuffer.this.workingBuffer.putShort(value);
        }

        @Override
        public void putInt(int value) {
            this.ensureEnoughSpace(4);
            WALBuffer.this.workingBuffer.putInt(value);
        }

        @Override
        public void putLong(long value) {
            this.ensureEnoughSpace(8);
            WALBuffer.this.workingBuffer.putLong(value);
        }

        @Override
        public void putFloat(float value) {
            this.ensureEnoughSpace(4);
            WALBuffer.this.workingBuffer.putFloat(value);
        }

        @Override
        public void putDouble(double value) {
            this.ensureEnoughSpace(8);
            WALBuffer.this.workingBuffer.putDouble(value);
        }

        @Override
        public int position() {
            return this.flushedBytesNum + WALBuffer.this.workingBuffer.position();
        }
    }
}

