/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.mailbox.cassandra.mail;

import com.google.common.base.Preconditions;
import java.security.SecureRandom;
import java.time.Duration;
import javax.inject.Inject;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.james.backends.cassandra.init.configuration.CassandraConfiguration;
import org.apache.james.backends.cassandra.init.configuration.JamesExecutionProfiles;
import org.apache.james.core.Username;
import org.apache.james.mailbox.acl.ACLDiff;
import org.apache.james.mailbox.cassandra.ids.CassandraId;
import org.apache.james.mailbox.cassandra.mail.ACLMapper;
import org.apache.james.mailbox.cassandra.mail.CassandraMailboxDAO;
import org.apache.james.mailbox.cassandra.mail.CassandraMailboxPathV3DAO;
import org.apache.james.mailbox.cassandra.mail.CassandraUserMailboxRightsDAO;
import org.apache.james.mailbox.cassandra.mail.task.SolveMailboxInconsistenciesService;
import org.apache.james.mailbox.exception.MailboxExistsException;
import org.apache.james.mailbox.exception.MailboxNotFoundException;
import org.apache.james.mailbox.model.Mailbox;
import org.apache.james.mailbox.model.MailboxACL;
import org.apache.james.mailbox.model.MailboxId;
import org.apache.james.mailbox.model.MailboxPath;
import org.apache.james.mailbox.model.UidValidity;
import org.apache.james.mailbox.model.search.MailboxQuery;
import org.apache.james.mailbox.store.mail.MailboxMapper;
import org.apache.james.util.FunctionalUtils;
import org.reactivestreams.Publisher;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.util.retry.Retry;

public class CassandraMailboxMapper
implements MailboxMapper {
    private static final int MAX_RETRY = 5;
    private static final Duration MIN_RETRY_BACKOFF = Duration.ofMillis(10L);
    private static final Duration MAX_RETRY_BACKOFF = Duration.ofMillis(1000L);
    private static final int CONCURRENCY = 10;
    private final CassandraMailboxDAO mailboxDAO;
    private final CassandraMailboxPathV3DAO mailboxPathV3DAO;
    private final ACLMapper aclMapper;
    private final CassandraUserMailboxRightsDAO userMailboxRightsDAO;
    private final CassandraConfiguration cassandraConfiguration;
    private final SecureRandom secureRandom;

    @Inject
    public CassandraMailboxMapper(CassandraMailboxDAO mailboxDAO, CassandraMailboxPathV3DAO mailboxPathV3DAO, CassandraUserMailboxRightsDAO userMailboxRightsDAO, ACLMapper aclMapper, CassandraConfiguration cassandraConfiguration) {
        this.mailboxDAO = mailboxDAO;
        this.mailboxPathV3DAO = mailboxPathV3DAO;
        this.userMailboxRightsDAO = userMailboxRightsDAO;
        this.aclMapper = aclMapper;
        this.cassandraConfiguration = cassandraConfiguration;
        this.secureRandom = new SecureRandom();
    }

    private Mono<Mailbox> performReadRepair(CassandraId id) {
        if (this.shouldReadRepair()) {
            return this.mailboxDAO.retrieveMailbox(id).flatMap(mailboxEntry -> SolveMailboxInconsistenciesService.Inconsistency.detectMailboxDaoInconsistency(mailboxEntry, this.mailboxPathV3DAO.retrieve(mailboxEntry.generateAssociatedPath(), JamesExecutionProfiles.ConsistencyChoice.STRONG)).flatMap(inconsistency -> inconsistency.fix(new SolveMailboxInconsistenciesService.Context(), this.mailboxDAO, this.mailboxPathV3DAO).then(Mono.just((Object)mailboxEntry))));
        }
        return this.mailboxDAO.retrieveMailbox(id);
    }

    private Mono<Mailbox> performReadRepair(MailboxPath path) {
        if (this.shouldReadRepair()) {
            return this.mailboxPathV3DAO.retrieve(path, JamesExecutionProfiles.ConsistencyChoice.STRONG).flatMap(this::performPathReadRepair);
        }
        return this.mailboxPathV3DAO.retrieve(path, this.consistencyChoice());
    }

    private JamesExecutionProfiles.ConsistencyChoice consistencyChoice() {
        if (this.cassandraConfiguration.isMailboxReadStrongConsistency()) {
            return JamesExecutionProfiles.ConsistencyChoice.STRONG;
        }
        return JamesExecutionProfiles.ConsistencyChoice.WEAK;
    }

    private Flux<Mailbox> performReadRepair(Flux<Mailbox> pathEntries) {
        return pathEntries.flatMap(mailboxPathEntry -> {
            if (this.shouldReadRepair()) {
                return this.performPathReadRepair((Mailbox)mailboxPathEntry);
            }
            return Mono.just((Object)mailboxPathEntry);
        }, 10);
    }

    private Mono<Mailbox> performPathReadRepair(Mailbox mailboxPathEntry) {
        return SolveMailboxInconsistenciesService.Inconsistency.detectMailboxPathDaoInconsistency(mailboxPathEntry, this.mailboxDAO.retrieveMailbox((CassandraId)mailboxPathEntry.getMailboxId())).flatMap(inconsistency -> inconsistency.fix(new SolveMailboxInconsistenciesService.Context(), this.mailboxDAO, this.mailboxPathV3DAO).then(Mono.just((Object)mailboxPathEntry)));
    }

    private boolean shouldReadRepair() {
        return this.cassandraConfiguration.getMailboxReadRepair() > 0.0f && this.secureRandom.nextFloat() < this.cassandraConfiguration.getMailboxReadRepair();
    }

    public Mono<Void> delete(Mailbox mailbox) {
        CassandraId mailboxId = (CassandraId)mailbox.getMailboxId();
        return this.deletePath(mailbox).thenEmpty((Publisher)this.mailboxDAO.delete(mailboxId).retryWhen((Retry)Retry.backoff((long)5L, (Duration)MIN_RETRY_BACKOFF).maxBackoff(MAX_RETRY_BACKOFF)));
    }

    private Mono<Void> deletePath(Mailbox mailbox) {
        return this.mailboxPathV3DAO.delete(mailbox.generateAssociatedPath());
    }

    public Mono<Mailbox> findMailboxByPath(MailboxPath path) {
        return this.performReadRepair(path).flatMap(this::addAcl);
    }

    private Mono<Mailbox> addAcl(Mailbox mailbox) {
        CassandraId mailboxId = (CassandraId)mailbox.getMailboxId();
        return this.aclMapper.getACL(mailboxId).map(acl -> {
            mailbox.setACL(acl);
            return mailbox;
        }).switchIfEmpty(Mono.just((Object)mailbox));
    }

    public Mono<Boolean> pathExists(MailboxPath path) {
        return this.performReadRepair(path).hasElement();
    }

    public Mono<Mailbox> findMailboxById(MailboxId id) {
        CassandraId mailboxId = (CassandraId)id;
        return this.retrieveMailbox(mailboxId).switchIfEmpty(Mono.error(() -> new MailboxNotFoundException(id)));
    }

    private Mono<Mailbox> retrieveMailbox(CassandraId mailboxId) {
        Mono<MailboxACL> acl = this.retrieveAcl(mailboxId);
        Mono<Mailbox> simpleMailbox = this.performReadRepair(mailboxId);
        return acl.zipWith(simpleMailbox, this::addAcl);
    }

    private Mono<MailboxACL> retrieveAcl(CassandraId mailboxId) {
        return this.aclMapper.getACL(mailboxId).defaultIfEmpty((Object)MailboxACL.EMPTY);
    }

    private Mailbox addAcl(MailboxACL acl, Mailbox mailbox) {
        mailbox.setACL(acl);
        return mailbox;
    }

    public Flux<Mailbox> findMailboxWithPathLike(MailboxQuery.UserBound query) {
        String fixedNamespace = query.getFixedNamespace();
        Username fixedUser = query.getFixedUser();
        return this.performReadRepair(this.listMailboxes(fixedNamespace, fixedUser)).filter(mailbox -> query.isPathMatch(mailbox.generateAssociatedPath())).flatMap(this::addAcl, 10);
    }

    private Flux<Mailbox> listMailboxes(String fixedNamespace, Username fixedUser) {
        return this.mailboxPathV3DAO.listUserMailboxes(fixedNamespace, fixedUser, this.consistencyChoice());
    }

    public Mono<Mailbox> create(MailboxPath mailboxPath, UidValidity uidValidity) {
        CassandraId cassandraId = CassandraId.timeBased();
        Mailbox mailbox = new Mailbox(mailboxPath, uidValidity, (MailboxId)cassandraId);
        return this.mailboxPathV3DAO.save(mailbox).filter(isCreated -> isCreated).flatMap(mailboxHasCreated -> this.persistMailboxEntity(mailbox).thenReturn((Object)mailbox)).switchIfEmpty(Mono.error(() -> new MailboxExistsException(mailbox.generateAssociatedPath().asString())));
    }

    public Mono<MailboxId> rename(Mailbox mailbox) {
        Preconditions.checkNotNull((Object)mailbox.getMailboxId(), (Object)"A mailbox we want to rename should have a defined mailboxId");
        CassandraId cassandraId = (CassandraId)mailbox.getMailboxId();
        return this.tryRename(mailbox, cassandraId).filter(FunctionalUtils.identityPredicate()).switchIfEmpty(Mono.error(() -> new MailboxExistsException(mailbox.generateAssociatedPath().asString()))).thenReturn((Object)cassandraId);
    }

    private Mono<Boolean> tryRename(Mailbox cassandraMailbox, CassandraId cassandraId) {
        return this.mailboxDAO.retrieveMailbox(cassandraId).flatMap(mailbox -> this.mailboxPathV3DAO.save(cassandraMailbox).filter(isCreated -> isCreated).flatMap(mailboxHasCreated -> this.deletePreviousMailboxPathReference(mailbox.generateAssociatedPath()).then(this.persistMailboxEntity(cassandraMailbox)).thenReturn((Object)true)).defaultIfEmpty((Object)false)).switchIfEmpty(Mono.error(() -> new MailboxNotFoundException((MailboxId)cassandraId)));
    }

    private Mono<Void> persistMailboxEntity(Mailbox cassandraMailbox) {
        return this.mailboxDAO.save(cassandraMailbox).retryWhen((Retry)Retry.backoff((long)5L, (Duration)MIN_RETRY_BACKOFF).maxBackoff(MAX_RETRY_BACKOFF));
    }

    private Mono<Void> deletePreviousMailboxPathReference(MailboxPath mailboxPath) {
        return this.mailboxPathV3DAO.delete(mailboxPath).retryWhen((Retry)Retry.backoff((long)5L, (Duration)MIN_RETRY_BACKOFF).maxBackoff(MAX_RETRY_BACKOFF));
    }

    public Mono<Boolean> hasChildren(Mailbox mailbox, char delimiter) {
        return this.performReadRepair(this.listMailboxes(mailbox.getNamespace(), mailbox.getUser())).filter(idAndPath -> this.isPathChildOfMailbox((Mailbox)idAndPath, mailbox, delimiter)).hasElements();
    }

    private boolean isPathChildOfMailbox(Mailbox candidate, Mailbox mailbox, char delimiter) {
        return candidate.generateAssociatedPath().getName().startsWith(mailbox.getName() + delimiter);
    }

    public Flux<Mailbox> list() {
        return this.performReadRepair(this.mailboxDAO.retrieveAllMailboxes()).flatMap(this::addAcl, 10);
    }

    public Mono<ACLDiff> updateACL(Mailbox mailbox, MailboxACL.ACLCommand mailboxACLCommand) {
        CassandraId cassandraId = (CassandraId)mailbox.getMailboxId();
        return this.aclMapper.updateACL(cassandraId, mailboxACLCommand);
    }

    public Mono<ACLDiff> setACL(Mailbox mailbox, MailboxACL mailboxACL) {
        CassandraId cassandraId = (CassandraId)mailbox.getMailboxId();
        return this.aclMapper.setACL(cassandraId, mailboxACL);
    }

    public Flux<Mailbox> findNonPersonalMailboxes(Username userName, MailboxACL.Right right) {
        return this.performReadRepair((Flux<Mailbox>)this.userMailboxRightsDAO.listRightsForUser(userName).filter(mailboxId -> ((MailboxACL.Rfc4314Rights)mailboxId.getRight()).contains(right)).map(Pair::getLeft).flatMap(this::retrieveMailbox, 10));
    }
}

