/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.server.security.delegation;

import com.google.common.base.Preconditions;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.accumulo.fate.zookeeper.ZooUtil;
import org.apache.accumulo.server.security.delegation.AuthenticationKey;
import org.apache.accumulo.server.zookeeper.ZooReaderWriter;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.data.ACL;
import org.apache.zookeeper.data.Id;
import org.apache.zookeeper.data.Stat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZooAuthenticationKeyDistributor {
    private static final Logger log = LoggerFactory.getLogger(ZooAuthenticationKeyDistributor.class);
    private final ZooReaderWriter zk;
    private final String baseNode;
    private AtomicBoolean initialized = new AtomicBoolean(false);

    public ZooAuthenticationKeyDistributor(ZooReaderWriter zk, String baseNode) {
        Objects.requireNonNull(zk);
        Objects.requireNonNull(baseNode);
        this.zk = zk;
        this.baseNode = baseNode;
    }

    public synchronized void initialize() throws KeeperException, InterruptedException {
        if (this.initialized.get()) {
            return;
        }
        if (!this.zk.exists(this.baseNode)) {
            if (!this.zk.putPrivatePersistentData(this.baseNode, new byte[0], ZooUtil.NodeExistsPolicy.FAIL)) {
                throw new AssertionError((Object)"Got false from putPrivatePersistentData method");
            }
        } else {
            List acls = this.zk.getACL(this.baseNode, new Stat());
            if (1 == acls.size()) {
                ACL actualAcl = (ACL)acls.get(0);
                ACL expectedAcl = (ACL)ZooUtil.PRIVATE.get(0);
                Id actualId = actualAcl.getId();
                if (actualAcl.getPerms() == expectedAcl.getPerms() && actualId.getScheme().equals("digest") && actualId.getId().startsWith("accumulo:")) {
                    this.initialized.set(true);
                    return;
                }
            } else {
                log.error("Saw more than one ACL on the node");
            }
            log.error("Expected {} to have ACLs {} but was {}", new Object[]{this.baseNode, ZooUtil.PRIVATE, acls});
            throw new IllegalStateException("Delegation token secret key node in ZooKeeper is not protected.");
        }
        this.initialized.set(true);
    }

    public List<AuthenticationKey> getCurrentKeys() throws KeeperException, InterruptedException {
        Preconditions.checkState((boolean)this.initialized.get(), (Object)"Not initialized");
        List children = this.zk.getChildren(this.baseNode);
        if (children.isEmpty()) {
            return Collections.emptyList();
        }
        ArrayList<AuthenticationKey> keys = new ArrayList<AuthenticationKey>(children.size());
        for (String child : children) {
            byte[] data = this.zk.getData(this.qualifyPath(child), null);
            if (null == data) continue;
            AuthenticationKey key = new AuthenticationKey();
            try {
                key.readFields(new DataInputStream(new ByteArrayInputStream(data)));
            }
            catch (IOException e) {
                throw new AssertionError("Error reading from in-memory buffer which should not happen", e);
            }
            keys.add(key);
        }
        return keys;
    }

    public synchronized void advertise(AuthenticationKey newKey) throws KeeperException, InterruptedException {
        Preconditions.checkState((boolean)this.initialized.get(), (Object)"Not initialized");
        Objects.requireNonNull(newKey);
        String path = this.qualifyPath(newKey);
        if (this.zk.exists(path)) {
            log.warn("AuthenticationKey with ID '{}' already exists in ZooKeeper", (Object)newKey.getKeyId());
            return;
        }
        ByteArrayOutputStream baos = new ByteArrayOutputStream(4096);
        try {
            newKey.write(new DataOutputStream(baos));
        }
        catch (IOException e) {
            throw new AssertionError("Should not get exception writing to in-memory buffer", e);
        }
        byte[] serializedKey = baos.toByteArray();
        log.debug("Advertising AuthenticationKey with keyId {} in ZooKeeper at {}", (Object)newKey.getKeyId(), (Object)path);
        this.zk.putPrivatePersistentData(path, serializedKey, ZooUtil.NodeExistsPolicy.FAIL);
    }

    public synchronized void remove(AuthenticationKey key) throws KeeperException, InterruptedException {
        Preconditions.checkState((boolean)this.initialized.get(), (Object)"Not initialized");
        Objects.requireNonNull(key);
        String path = this.qualifyPath(key);
        if (!this.zk.exists(path)) {
            log.warn("AuthenticationKey with ID '{}' doesn't exist in ZooKeeper", (Object)key.getKeyId());
            return;
        }
        log.debug("Removing AuthenticationKey with keyId {} from ZooKeeper at {}", (Object)key.getKeyId(), (Object)path);
        this.zk.delete(path, -1);
    }

    String qualifyPath(String keyId) {
        return this.baseNode + "/" + keyId;
    }

    String qualifyPath(AuthenticationKey key) {
        return this.qualifyPath(Integer.toString(key.getKeyId()));
    }
}

