/*
 * Decompiled with CFR 0.152.
 */
package org.apache.uniffle.server;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.RandomUtils;
import org.apache.uniffle.common.StorageType;
import org.apache.uniffle.common.config.RssConf;
import org.apache.uniffle.common.util.RssUtils;
import org.apache.uniffle.guava.annotations.VisibleForTesting;
import org.apache.uniffle.guava.collect.Lists;
import org.apache.uniffle.server.Checker;
import org.apache.uniffle.server.ShuffleServerConf;
import org.apache.uniffle.server.ShuffleServerMetrics;
import org.apache.uniffle.storage.common.LocalStorage;
import org.apache.uniffle.storage.util.ShuffleStorageUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LocalStorageChecker
extends Checker {
    private static final Logger LOG = LoggerFactory.getLogger(LocalStorageChecker.class);
    public static final String CHECKER_DIR_NAME = ".check";
    private final double diskMaxUsagePercentage;
    private final double diskRecoveryUsagePercentage;
    private final double minStorageHealthyPercentage;
    private final List<StorageInfo> storageInfos = Lists.newArrayList();
    private boolean isHealthy = true;

    public LocalStorageChecker(ShuffleServerConf conf, List<LocalStorage> storages) {
        super(conf);
        List basePaths = RssUtils.getConfiguredLocalDirs((RssConf)conf);
        if (CollectionUtils.isEmpty((Collection)basePaths)) {
            throw new IllegalArgumentException("The base path cannot be empty");
        }
        String storageType = ((StorageType)conf.get(ShuffleServerConf.RSS_STORAGE_TYPE)).name();
        if (!ShuffleStorageUtils.containsLocalFile((String)storageType)) {
            throw new IllegalArgumentException("Only StorageType contains LOCALFILE support storageChecker");
        }
        for (LocalStorage storage : storages) {
            this.storageInfos.add(new StorageInfo(storage));
        }
        this.diskMaxUsagePercentage = conf.getDouble(ShuffleServerConf.HEALTH_STORAGE_MAX_USAGE_PERCENTAGE);
        this.diskRecoveryUsagePercentage = conf.getDouble(ShuffleServerConf.HEALTH_STORAGE_RECOVERY_USAGE_PERCENTAGE);
        this.minStorageHealthyPercentage = conf.getDouble(ShuffleServerConf.HEALTH_MIN_STORAGE_PERCENTAGE);
    }

    @Override
    public boolean checkIsHealthy() {
        AtomicInteger num = new AtomicInteger(0);
        AtomicLong totalSpace = new AtomicLong(0L);
        AtomicLong wholeDiskUsedSpace = new AtomicLong(0L);
        AtomicLong serviceUsedSpace = new AtomicLong(0L);
        AtomicInteger corruptedDirs = new AtomicInteger(0);
        CountDownLatch cdl = new CountDownLatch(this.storageInfos.size());
        this.storageInfos.parallelStream().forEach(storageInfo -> {
            if (!storageInfo.checkStorageReadAndWrite()) {
                storageInfo.markCorrupted();
                corruptedDirs.incrementAndGet();
                cdl.countDown();
                return;
            }
            totalSpace.addAndGet(this.getTotalSpace(((StorageInfo)storageInfo).storageDir));
            wholeDiskUsedSpace.addAndGet(this.getWholeDiskUsedSpace(((StorageInfo)storageInfo).storageDir));
            serviceUsedSpace.addAndGet(LocalStorageChecker.getServiceUsedSpace(((StorageInfo)storageInfo).storageDir));
            if (storageInfo.checkIsSpaceEnough()) {
                num.incrementAndGet();
            }
            cdl.countDown();
        });
        try {
            cdl.await();
        }
        catch (InterruptedException e) {
            LOG.error("Failed to check local storage!");
        }
        ShuffleServerMetrics.gaugeLocalStorageTotalSpace.set((double)totalSpace.get());
        ShuffleServerMetrics.gaugeLocalStorageWholeDiskUsedSpace.set((double)wholeDiskUsedSpace.get());
        ShuffleServerMetrics.gaugeLocalStorageServiceUsedSpace.set((double)serviceUsedSpace.get());
        ShuffleServerMetrics.gaugeLocalStorageTotalDirsNum.set((double)this.storageInfos.size());
        ShuffleServerMetrics.gaugeLocalStorageCorruptedDirsNum.set((double)corruptedDirs.get());
        ShuffleServerMetrics.gaugeLocalStorageUsedSpaceRatio.set((double)wholeDiskUsedSpace.get() * 1.0 / (double)totalSpace.get());
        if (this.storageInfos.isEmpty()) {
            if (this.isHealthy) {
                LOG.info("shuffle server become unhealthy because of empty storage");
            }
            this.isHealthy = false;
            return false;
        }
        double availablePercentage = (double)num.get() * 100.0 / (double)this.storageInfos.size();
        if (Double.compare(availablePercentage, this.minStorageHealthyPercentage) >= 0) {
            if (!this.isHealthy) {
                LOG.info("shuffle server become healthy");
            }
            this.isHealthy = true;
        } else {
            if (this.isHealthy) {
                LOG.info("shuffle server become unhealthy");
            }
            this.isHealthy = false;
        }
        return this.isHealthy;
    }

    @VisibleForTesting
    long getTotalSpace(File file) {
        return file.getTotalSpace();
    }

    @VisibleForTesting
    long getWholeDiskUsedSpace(File file) {
        return file.getTotalSpace() - file.getUsableSpace();
    }

    protected static long getServiceUsedSpace(File storageDir) {
        if (storageDir == null || !storageDir.exists()) {
            return 0L;
        }
        if (storageDir.isFile()) {
            return storageDir.length();
        }
        File[] files = storageDir.listFiles();
        if (files == null) {
            return 0L;
        }
        long totalUsage = 0L;
        for (File file : files) {
            if (file.isFile()) {
                totalUsage += file.length();
                continue;
            }
            totalUsage += LocalStorageChecker.getServiceUsedSpace(file);
        }
        return totalUsage;
    }

    class StorageInfo {
        private final File storageDir;
        private final LocalStorage storage;
        private boolean isHealthy;

        StorageInfo(LocalStorage storage) {
            this.storageDir = new File(storage.getBasePath());
            this.isHealthy = true;
            this.storage = storage;
        }

        boolean checkIsSpaceEnough() {
            if (Double.compare(0.0, LocalStorageChecker.this.getTotalSpace(this.storageDir)) == 0) {
                this.isHealthy = false;
                return false;
            }
            double usagePercent = (double)LocalStorageChecker.this.getWholeDiskUsedSpace(this.storageDir) * 100.0 / (double)LocalStorageChecker.this.getTotalSpace(this.storageDir);
            if (this.isHealthy) {
                if (Double.compare(usagePercent, LocalStorageChecker.this.diskMaxUsagePercentage) >= 0) {
                    this.isHealthy = false;
                    LOG.info("storage {} become unhealthy", (Object)this.storageDir.getAbsolutePath());
                }
            } else if (Double.compare(usagePercent, LocalStorageChecker.this.diskRecoveryUsagePercentage) <= 0) {
                this.isHealthy = true;
                LOG.info("storage {} become healthy", (Object)this.storageDir.getAbsolutePath());
            }
            return this.isHealthy;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        boolean checkStorageReadAndWrite() {
            if (this.storage.isCorrupted()) {
                return false;
            }
            File checkDir = new File(this.storageDir, LocalStorageChecker.CHECKER_DIR_NAME);
            try {
                if (!checkDir.mkdirs()) {
                    boolean bl = false;
                    return bl;
                }
                File writeFile = new File(checkDir, "test");
                if (!writeFile.createNewFile()) {
                    boolean ioe = false;
                    return ioe;
                }
                byte[] data = RandomUtils.nextBytes((int)1024);
                try (FileOutputStream fos = new FileOutputStream(writeFile);){
                    fos.write(data);
                    fos.flush();
                    fos.getFD().sync();
                }
                byte[] readData = new byte[1024];
                int readBytes = -1;
                try (FileInputStream fis = new FileInputStream(writeFile);){
                    int hasReadBytes = 0;
                    do {
                        readBytes = fis.read(readData);
                        if (hasReadBytes < 1024) {
                            for (int i = 0; i < readBytes; ++i) {
                                if (data[hasReadBytes + i] == readData[i]) continue;
                                boolean bl = false;
                                return bl;
                            }
                        }
                        hasReadBytes += readBytes;
                    } while (readBytes != -1);
                }
            }
            catch (Exception e) {
                LOG.error("Storage read and write error. Storage dir: {}", (Object)this.storageDir, (Object)e);
                if (e.getMessage() != null && "No space left on device".equals(e.getMessage())) {
                    boolean bl = true;
                    return bl;
                }
                boolean bl = false;
                return bl;
            }
            finally {
                try {
                    FileUtils.deleteDirectory((File)checkDir);
                }
                catch (IOException ioe) {
                    LOG.error("delete directory fail. Storage dir: {}", (Object)this.storageDir, (Object)ioe);
                    return false;
                }
            }
            return true;
        }

        public void markCorrupted() {
            this.storage.markCorrupted();
        }
    }
}

