#ifndef IGCC_SPIRV_EXTENSIONS_SUPPORT_H
#define IGCC_SPIRV_EXTENSIONS_SUPPORT_H

#include <vector>
#include <string>
#include "igfxfmid.h"

namespace IGC {
namespace SPIRVExtensionsSupport {

// Helper function for core family hierarchy checks
inline bool isCoreChildOf(PLATFORM Platform, GFXCORE_FAMILY Core) {
  return Platform.eRenderCoreFamily >= Core;
}

// Helper function for product family hierarchy checks
inline bool isProductChildOf(PLATFORM Platform, PRODUCT_FAMILY Product) {
  return Platform.eProductFamily >= Product;
}

// SPIR-V Extension and Capability structures
struct SPIRVCapability {
  std::string Name;
};

struct SPIRVExtension {
  std::string Name;
  std::string SpecURL;
  std::vector<SPIRVCapability> Capabilities;
};

// Forward declarations
inline bool isExtensionSupported(const std::string& ExtensionName, PLATFORM Platform, bool includeExperimental);
inline bool isCapabilitySupported(const std::string& CapabilityName, PLATFORM Platform, bool includeExperimental);

// Static vector of all defined extensions with their capabilities
static const std::vector<SPIRVExtension> AllExtensions = {
  {
    "SPV_EXT_float8",
    "https://github.khronos.org/SPIRV-Registry/extensions/EXT/SPV_EXT_float8.html",
    {
      {"Float8EXT"},
      {"Float8CooperativeMatrixEXT"}
    }
  },
  {
    "SPV_EXT_optnone",
    "https://github.khronos.org/SPIRV-Registry/extensions/EXT/SPV_EXT_optnone.html",
    {
      {"OptNoneEXT"}
    }
  },
  {
    "SPV_EXT_shader_atomic_float16_add",
    "https://github.khronos.org/SPIRV-Registry/extensions/EXT/SPV_EXT_shader_atomic_float16_add.html",
    {
      {"AtomicFloat16AddEXT"}
    }
  },
  {
    "SPV_EXT_shader_atomic_float_add",
    "https://github.khronos.org/SPIRV-Registry/extensions/EXT/SPV_EXT_shader_atomic_float_add.html",
    {
      {"AtomicFloat32AddEXT"},
      {"AtomicFloat64AddEXT"}
    }
  },
  {
    "SPV_EXT_shader_atomic_float_min_max",
    "https://github.khronos.org/SPIRV-Registry/extensions/EXT/SPV_EXT_shader_atomic_float_min_max.html",
    {
      {"AtomicFloat16MinMaxEXT"},
      {"AtomicFloat32MinMaxEXT"},
      {"AtomicFloat64MinMaxEXT"}
    }
  },
  {
    "SPV_INTEL_16bit_atomics",
    "https://github.com/intel/llvm/pull/20009",
    {
      {"AtomicInt16CompareExchangeINTEL"},
      {"Int16AtomicsINTEL"},
      {"AtomicBFloat16LoadStoreINTEL"},
      {"AtomicBFloat16AddINTEL"},
      {"AtomicBFloat16MinMaxINTEL"}
    }
  },
  {
    "SPV_INTEL_2d_block_io",
    "https://github.com/KhronosGroup/SPIRV-Registry/blob/main/extensions/INTEL/SPV_INTEL_2d_block_io.asciidoc",
    {
      {"Subgroup2DBlockIOINTEL"},
      {"Subgroup2DBlockTransformINTEL"},
      {"Subgroup2DBlockTransposeINTEL"}
    }
  },
  {
    "SPV_INTEL_arbitrary_precision_integers",
    "https://github.com/KhronosGroup/SPIRV-Registry/blob/main/extensions/INTEL/SPV_INTEL_arbitrary_precision_integers.html",
    {
      {"ArbitraryPrecisionIntegersINTEL"}
    }
  },
  {
    "SPV_INTEL_bfloat16_arithmetic",
    "https://github.com/intel/llvm/blob/sycl/sycl/doc/design/spirv-extensions/SPV_INTEL_bfloat16_arithmetic.asciidoc",
    {
      {"BFloat16ArithmeticINTEL"}
    }
  },
  {
    "SPV_INTEL_bfloat16_conversion",
    "https://github.khronos.org/SPIRV-Registry/extensions/INTEL/SPV_INTEL_bfloat16_conversion.html",
    {
      {"BFloat16ConversionINTEL"}
    }
  },
  {
    "SPV_INTEL_bindless_images",
    "https://github.khronos.org/SPIRV-Registry/extensions/INTEL/SPV_INTEL_bindless_images.html",
    {
      {"BindlessImagesINTEL"}
    }
  },
  {
    "SPV_INTEL_cache_controls",
    "https://github.com/KhronosGroup/SPIRV-Registry/blob/main/extensions/INTEL/SPV_INTEL_cache_controls.asciidoc",
    {
      {"CacheControlsINTEL"}
    }
  },
  {
    "SPV_INTEL_debug_module",
    "https://github.com/intel/llvm/blob/sycl/sycl/doc/design/spirv-extensions/SPV_INTEL_debug_module.asciidoc",
    {
      {"DebugInfoModuleINTEL"}
    }
  },
  {
    "SPV_INTEL_float4",
    "https://github.com/intel/llvm/pull/20467",
    {
      {"Float4TypeINTEL"},
      {"Float4CooperativeMatrixINTEL"}
    }
  },
  {
    "SPV_INTEL_fp_conversions",
    "https://github.com/intel/llvm/pull/20467",
    {
      {"FloatConversionsINTEL"}
    }
  },
  {
    "SPV_INTEL_fp_fast_math_mode",
    "https://github.khronos.org/SPIRV-Registry/extensions/INTEL/SPV_INTEL_fp_fast_math_mode.html",
    {
      {"FPFastMathModeINTEL"}
    }
  },
  {
    "SPV_INTEL_fp_max_error",
    "https://github.khronos.org/SPIRV-Registry/extensions/INTEL/SPV_INTEL_fp_max_error.html",
    {
      {"FPMaxErrorINTEL"}
    }
  },
  {
    "SPV_INTEL_function_pointers",
    "https://github.com/intel/llvm/blob/sycl/sycl/doc/design/spirv-extensions/SPV_INTEL_function_pointers.asciidoc",
    {
      {"FunctionPointersINTEL"},
      {"IndirectReferencesINTEL"}
    }
  },
  {
    "SPV_INTEL_global_variable_decorations",
    "https://github.khronos.org/SPIRV-Registry/extensions/INTEL/SPV_INTEL_global_variable_host_access.html",
    {
    }
  },
  {
    "SPV_INTEL_global_variable_host_access",
    "https://github.khronos.org/SPIRV-Registry/extensions/INTEL/SPV_INTEL_global_variable_host_access.html",
    {
      {"GlobalVariableHostAccessINTEL"}
    }
  },
  {
    "SPV_INTEL_inline_assembly",
    "https://github.com/intel/llvm/blob/sycl/sycl/doc/design/spirv-extensions/SPV_INTEL_inline_assembly.asciidoc",
    {
      {"AsmINTEL"}
    }
  },
  {
    "SPV_INTEL_int4",
    "https://github.com/KhronosGroup/SPIRV-Registry/blob/main/extensions/INTEL/SPV_INTEL_int4.asciidoc",
    {
      {"Int4TypeINTEL"},
      {"Int4CooperativeMatrixINTEL"}
    }
  },
  {
    "SPV_INTEL_joint_matrix",
    "https://github.com/intel/llvm/blob/sycl/sycl/doc/design/spirv-extensions/SPV_INTEL_joint_matrix.asciidoc",
    {
      {"PackedCooperativeMatrixINTEL"},
      {"CooperativeMatrixInvocationInstructionsINTEL"},
      {"CooperativeMatrixTF32ComponentTypeINTEL"},
      {"CooperativeMatrixBFloat16ComponentTypeINTEL"},
      {"CooperativeMatrixPrefetchINTEL"}
    }
  },
  {
    "SPV_INTEL_kernel_attributes",
    "https://github.khronos.org/SPIRV-Registry/extensions/INTEL/SPV_INTEL_kernel_attributes.html",
    {
      {"KernelAttributesINTEL"},
      {"FPGAKernelAttributesINTEL"},
      {"FPGAKernelAttributesv2INTEL"}
    }
  },
  {
    "SPV_INTEL_long_composites",
    "https://github.khronos.org/SPIRV-Registry/extensions/INTEL/SPV_INTEL_long_composites.html",
    {
      {"CapabilityLongCompositesINTEL"}
    }
  },
  {
    "SPV_INTEL_media_block_io",
    "https://github.khronos.org/SPIRV-Registry/extensions/INTEL/SPV_INTEL_media_block_io.html",
    {
      {"SubgroupImageMediaBlockIOINTEL"}
    }
  },
  {
    "SPV_INTEL_memory_access_aliasing",
    "https://github.com/intel/llvm/pull/3426/files",
    {
      {"MemoryAccessAliasingINTEL"}
    }
  },
  {
    "SPV_INTEL_optnone",
    "https://github.khronos.org/SPIRV-Registry/extensions/INTEL/SPV_INTEL_optnone.html",
    {
      {"OptNoneINTEL"}
    }
  },
  {
    "SPV_INTEL_predicated_io",
    "https://github.com/intel/llvm/blob/sycl/sycl/doc/design/spirv-extensions/SPV_INTEL_predicated_io.asciidoc",
    {
      {"PredicatedIOINTEL"}
    }
  },
  {
    "SPV_INTEL_sigmoid",
    "https://github.com/intel/llvm/blob/sycl/sycl/doc/design/spirv-extensions/SPV_INTEL_sigmoid.asciidoc",
    {
      {"SigmoidINTEL"}
    }
  },
  {
    "SPV_INTEL_split_barrier",
    "https://github.khronos.org/SPIRV-Registry/extensions/INTEL/SPV_INTEL_split_barrier.html",
    {
      {"SplitBarrierINTEL"}
    }
  },
  {
    "SPV_INTEL_subgroup_buffer_prefetch",
    "https://github.khronos.org/SPIRV-Registry/extensions/INTEL/SPV_INTEL_subgroup_buffer_prefetch.html",
    {
      {"SubgroupBufferPrefetchINTEL"}
    }
  },
  {
    "SPV_INTEL_subgroup_matrix_multiply_accumulate",
    "https://github.khronos.org/SPIRV-Registry/extensions/INTEL/SPV_INTEL_subgroup_matrix_multiply_accumulate.html",
    {
      {"SubgroupMatrixMultiplyAccumulateINTEL"}
    }
  },
  {
    "SPV_INTEL_subgroup_scaled_matrix_multiply_accumulate",
    "https://github.com/intel/llvm/pull/20656",
    {
      {"Subgroup​Scaled​Matrix​Multiply​Accumulate​INTEL"}
    }
  },
  {
    "SPV_INTEL_subgroups",
    "https://github.khronos.org/SPIRV-Registry/extensions/INTEL/SPV_INTEL_subgroups.html",
    {
      {"SubgroupShuffleINTEL"},
      {"SubgroupBufferBlockIOINTEL"},
      {"SubgroupImageBlockIOINTEL"}
    }
  },
  {
    "SPV_INTEL_tensor_float32_conversion",
    "https://github.khronos.org/SPIRV-Registry/extensions/INTEL/SPV_INTEL_tensor_float32_conversion.html",
    {
      {"TensorFloat32RoundingINTEL"}
    }
  },
  {
    "SPV_INTEL_unstructured_loop_controls",
    "https://github.khronos.org/SPIRV-Registry/extensions/INTEL/SPV_INTEL_unstructured_loop_controls.html",
    {
      {"UnstructuredLoopControlsINTEL"}
    }
  },
  {
    "SPV_INTEL_variable_length_array",
    "https://github.khronos.org/SPIRV-Registry/extensions/INTEL/SPV_INTEL_variable_length_array.html",
    {
      {"VariableLengthArrayINTEL"},
      {"UntypedVariableLengthArrayINTEL"}
    }
  },
  {
    "SPV_INTEL_vector_compute",
    "https://github.com/intel/llvm/pull/1612",
    {
      {"VectorComputeINTEL"},
      {"VectorAnyINTEL"}
    }
  },
  {
    "SPV_KHR_bfloat16",
    "https://github.khronos.org/SPIRV-Registry/extensions/KHR/SPV_KHR_bfloat16.html",
    {
      {"BFloat16TypeKHR"},
      {"BFloat16DotProductKHR"},
      {"BFloat16CooperativeMatrixKHR"}
    }
  },
  {
    "SPV_KHR_bit_instructions",
    "https://github.khronos.org/SPIRV-Registry/extensions/KHR/SPV_KHR_bit_instructions.html",
    {
      {"BitInstructions"}
    }
  },
  {
    "SPV_KHR_cooperative_matrix",
    "https://github.khronos.org/SPIRV-Registry/extensions/KHR/SPV_KHR_cooperative_matrix.html",
    {
      {"CooperativeMatrixKHR"}
    }
  },
  {
    "SPV_KHR_expect_assume",
    "https://github.khronos.org/SPIRV-Registry/extensions/KHR/SPV_KHR_expect_assume.html",
    {
      {"ExpectAssumeKHR"}
    }
  },
  {
    "SPV_KHR_integer_dot_product",
    "https://github.khronos.org/SPIRV-Registry/extensions/KHR/SPV_KHR_integer_dot_product.html",
    {
      {"DotProductKHR"},
      {"DotProductInputAllKHR"},
      {"DotProductInput4x8BitKHR"},
      {"DotProductInput4x8BitPackedKHR"}
    }
  },
  {
    "SPV_KHR_linkonce_odr",
    "https://github.khronos.org/SPIRV-Registry/extensions/KHR/SPV_KHR_linkonce_odr.html",
    {
      {"Linkage"}
    }
  },
  {
    "SPV_KHR_no_integer_wrap_decoration",
    "https://github.khronos.org/SPIRV-Registry/extensions/KHR/SPV_KHR_no_integer_wrap_decoration.html",
    {
    }
  },
  {
    "SPV_KHR_non_semantic_info",
    "https://github.khronos.org/SPIRV-Registry/extensions/KHR/SPV_KHR_non_semantic_info.html",
    {
    }
  },
  {
    "SPV_KHR_shader_clock",
    "https://github.khronos.org/SPIRV-Registry/extensions/KHR/SPV_KHR_shader_clock.html",
    {
      {"ShaderClockKHR"}
    }
  },
  {
    "SPV_KHR_uniform_group_instructions",
    "https://github.khronos.org/SPIRV-Registry/extensions/KHR/SPV_KHR_uniform_group_instructions.html",
    {
      {"GroupUniformArithmeticKHR"}
    }
  }
};

// Individual extension/capability query functions
inline bool isExtensionSupported(const std::string& ExtensionName, PLATFORM Platform, bool includeExperimental) {
  if (ExtensionName == "SPV_EXT_float8") {
    if (includeExperimental) {
      if (isCoreChildOf(Platform, IGFX_XE3P_CORE)) return true;
    }
  }
  if (ExtensionName == "SPV_EXT_optnone") {
    if (true /* Supported on all platforms */) return true;
  }
  if (ExtensionName == "SPV_EXT_shader_atomic_float16_add") {
    if (isCoreChildOf(Platform, IGFX_XE3P_CORE)) return true;
  }
  if (ExtensionName == "SPV_EXT_shader_atomic_float_add") {
    if (isCapabilitySupported("AtomicFloat32AddEXT", Platform, includeExperimental) || isCapabilitySupported("AtomicFloat64AddEXT", Platform, includeExperimental)) return true;
  }
  if (ExtensionName == "SPV_EXT_shader_atomic_float_min_max") {
    if (isCapabilitySupported("AtomicFloat16MinMaxEXT", Platform, includeExperimental) || isCapabilitySupported("AtomicFloat32MinMaxEXT", Platform, includeExperimental) || isCapabilitySupported("AtomicFloat64MinMaxEXT", Platform, includeExperimental)) return true;
  }
  if (ExtensionName == "SPV_INTEL_16bit_atomics") {
    if (includeExperimental) {
      if (isCapabilitySupported("AtomicInt16CompareExchangeINTEL", Platform, includeExperimental) || isCapabilitySupported("Int16AtomicsINTEL", Platform, includeExperimental) || isCapabilitySupported("AtomicBFloat16LoadStoreINTEL", Platform, includeExperimental) || isCapabilitySupported("AtomicBFloat16AddINTEL", Platform, includeExperimental) || isCapabilitySupported("AtomicBFloat16MinMaxINTEL", Platform, includeExperimental)) return true;
    }
  }
  if (ExtensionName == "SPV_INTEL_2d_block_io") {
    if (isCoreChildOf(Platform, IGFX_XE_HPC_CORE)) return true;
  }
  if (ExtensionName == "SPV_INTEL_arbitrary_precision_integers") {
    if (includeExperimental) {
      if (true /* Supported on all platforms */) return true;
    }
  }
  if (ExtensionName == "SPV_INTEL_bfloat16_arithmetic") {
    if (includeExperimental) {
      if ((isCoreChildOf(Platform, IGFX_XE_HPG_CORE)) && (!(Platform.eProductFamily == IGFX_METEORLAKE))) return true;
    }
  }
  if (ExtensionName == "SPV_INTEL_bfloat16_conversion") {
    if ((isCoreChildOf(Platform, IGFX_XE_HPG_CORE)) && (!(Platform.eProductFamily == IGFX_METEORLAKE))) return true;
  }
  if (ExtensionName == "SPV_INTEL_bindless_images") {
    if (includeExperimental) {
      if ((isCoreChildOf(Platform, IGFX_XE_HPG_CORE)) && (!(Platform.eProductFamily == IGFX_PVC)) && (!(Platform.eProductFamily == IGFX_CRI))) return true;
    }
  }
  if (ExtensionName == "SPV_INTEL_cache_controls") {
    if (true /* Supported on all platforms */) return true;
  }
  if (ExtensionName == "SPV_INTEL_debug_module") {
    if (includeExperimental) {
      if (true /* Supported on all platforms */) return true;
    }
  }
  if (ExtensionName == "SPV_INTEL_float4") {
    if (includeExperimental) {
      if (isCoreChildOf(Platform, IGFX_XE3P_CORE)) return true;
    }
  }
  if (ExtensionName == "SPV_INTEL_fp_conversions") {
    if (includeExperimental) {
      if (isCoreChildOf(Platform, IGFX_XE3P_CORE)) return true;
    }
  }
  if (ExtensionName == "SPV_INTEL_fp_fast_math_mode") {
    if (includeExperimental) {
      if (true /* Supported on all platforms */) return true;
    }
  }
  if (ExtensionName == "SPV_INTEL_fp_max_error") {
    if (includeExperimental) {
      if (true /* Supported on all platforms */) return true;
    }
  }
  if (ExtensionName == "SPV_INTEL_function_pointers") {
    if (includeExperimental) {
      if (true /* Supported on all platforms */) return true;
    }
  }
  if (ExtensionName == "SPV_INTEL_global_variable_decorations") {
    if (includeExperimental) {
      if (true /* Supported on all platforms */) return true;
    }
  }
  if (ExtensionName == "SPV_INTEL_global_variable_host_access") {
    if (true /* Supported on all platforms */) return true;
  }
  if (ExtensionName == "SPV_INTEL_inline_assembly") {
    if (includeExperimental) {
      if (true /* Supported on all platforms */) return true;
    }
  }
  if (ExtensionName == "SPV_INTEL_int4") {
    if (includeExperimental) {
      if (isCapabilitySupported("Int4TypeINTEL", Platform, includeExperimental) || isCapabilitySupported("Int4CooperativeMatrixINTEL", Platform, includeExperimental)) return true;
    }
  }
  if (ExtensionName == "SPV_INTEL_joint_matrix") {
    if (includeExperimental) {
      if (isCapabilitySupported("PackedCooperativeMatrixINTEL", Platform, includeExperimental) || isCapabilitySupported("CooperativeMatrixInvocationInstructionsINTEL", Platform, includeExperimental) || isCapabilitySupported("CooperativeMatrixTF32ComponentTypeINTEL", Platform, includeExperimental) || isCapabilitySupported("CooperativeMatrixBFloat16ComponentTypeINTEL", Platform, includeExperimental) || isCapabilitySupported("CooperativeMatrixPrefetchINTEL", Platform, includeExperimental)) return true;
    }
  }
  if (ExtensionName == "SPV_INTEL_kernel_attributes") {
    if (includeExperimental) {
      if (isCapabilitySupported("KernelAttributesINTEL", Platform, includeExperimental) || isCapabilitySupported("FPGAKernelAttributesINTEL", Platform, includeExperimental) || isCapabilitySupported("FPGAKernelAttributesv2INTEL", Platform, includeExperimental)) return true;
    }
  }
  if (ExtensionName == "SPV_INTEL_long_composites") {
    if (true /* Supported on all platforms */) return true;
  }
  if (ExtensionName == "SPV_INTEL_media_block_io") {
    if ((Platform.eRenderCoreFamily == IGFX_GEN12LP_CORE) || (Platform.eRenderCoreFamily == IGFX_XE_HPG_CORE)) return true;
  }
  if (ExtensionName == "SPV_INTEL_memory_access_aliasing") {
    if (includeExperimental) {
      if (true /* Supported on all platforms */) return true;
    }
  }
  if (ExtensionName == "SPV_INTEL_optnone") {
    if (includeExperimental) {
      if (true /* Supported on all platforms */) return true;
    }
  }
  if (ExtensionName == "SPV_INTEL_predicated_io") {
    if (includeExperimental) {
      if (true /* Supported on all platforms */) return true;
    }
  }
  if (ExtensionName == "SPV_INTEL_sigmoid") {
    if (includeExperimental) {
      if (true /* Supported on all platforms */) return true;
    }
  }
  if (ExtensionName == "SPV_INTEL_split_barrier") {
    if (true /* Supported on all platforms */) return true;
  }
  if (ExtensionName == "SPV_INTEL_subgroup_buffer_prefetch") {
    if (true /* Supported on all platforms */) return true;
  }
  if (ExtensionName == "SPV_INTEL_subgroup_matrix_multiply_accumulate") {
    if ((isCoreChildOf(Platform, IGFX_XE_HPG_CORE)) && (!(Platform.eProductFamily == IGFX_METEORLAKE)) && (!(Platform.eProductFamily == IGFX_ARROWLAKE))) return true;
  }
  if (ExtensionName == "SPV_INTEL_subgroup_scaled_matrix_multiply_accumulate") {
    if (includeExperimental) {
      if (isCoreChildOf(Platform, IGFX_XE3P_CORE)) return true;
    }
  }
  if (ExtensionName == "SPV_INTEL_subgroups") {
    if (true /* Supported on all platforms */) return true;
  }
  if (ExtensionName == "SPV_INTEL_tensor_float32_conversion") {
    if (isCoreChildOf(Platform, IGFX_XE_HPC_CORE)) return true;
  }
  if (ExtensionName == "SPV_INTEL_unstructured_loop_controls") {
    if (true /* Supported on all platforms */) return true;
  }
  if (ExtensionName == "SPV_INTEL_variable_length_array") {
    if (includeExperimental) {
      if (true /* Supported on all platforms */) return true;
    }
  }
  if (ExtensionName == "SPV_INTEL_vector_compute") {
    if (includeExperimental) {
      if (true /* Supported on all platforms */) return true;
    }
  }
  if (ExtensionName == "SPV_KHR_bfloat16") {
    if (includeExperimental) {
      if ((isCoreChildOf(Platform, IGFX_XE_HPG_CORE)) && (!(Platform.eProductFamily == IGFX_METEORLAKE))) return true;
    }
  }
  if (ExtensionName == "SPV_KHR_bit_instructions") {
    if (true /* Supported on all platforms */) return true;
  }
  if (ExtensionName == "SPV_KHR_cooperative_matrix") {
    if (includeExperimental) {
      if (isCoreChildOf(Platform, IGFX_XE_HPG_CORE)) return true;
    }
  }
  if (ExtensionName == "SPV_KHR_expect_assume") {
    if (true /* Supported on all platforms */) return true;
  }
  if (ExtensionName == "SPV_KHR_integer_dot_product") {
    if (true /* Supported on all platforms */) return true;
  }
  if (ExtensionName == "SPV_KHR_linkonce_odr") {
    if (true /* Supported on all platforms */) return true;
  }
  if (ExtensionName == "SPV_KHR_no_integer_wrap_decoration") {
    if (true /* Supported on all platforms */) return true;
  }
  if (ExtensionName == "SPV_KHR_non_semantic_info") {
    if (includeExperimental) {
      if (true /* Supported on all platforms */) return true;
    }
  }
  if (ExtensionName == "SPV_KHR_shader_clock") {
    if (true /* Supported on all platforms */) return true;
  }
  if (ExtensionName == "SPV_KHR_uniform_group_instructions") {
    if (includeExperimental) {
      if (true /* Supported on all platforms */) return true;
    }
  }
  return false;
}

inline bool isCapabilitySupported(const std::string& CapabilityName, PLATFORM Platform, bool includeExperimental) {
  if (CapabilityName == "Float8EXT") {
    if (includeExperimental) {
      if (isCoreChildOf(Platform, IGFX_XE3P_CORE)) return true;
    }
  }
  if (CapabilityName == "Float8CooperativeMatrixEXT") {
    if (includeExperimental) {
      if (isCoreChildOf(Platform, IGFX_XE3P_CORE)) return true;
    }
  }
  if (CapabilityName == "OptNoneEXT") {
    if (true /* Supported on all platforms */) return true;
  }
  if (CapabilityName == "AtomicFloat16AddEXT") {
    if (isCoreChildOf(Platform, IGFX_XE3P_CORE)) return true;
  }
  if (CapabilityName == "AtomicFloat32AddEXT") {
    if (true /* Supported on all platforms */) return true;
  }
  if (CapabilityName == "AtomicFloat64AddEXT") {
    if (isProductChildOf(Platform, IGFX_METEORLAKE)) return true;
  }
  if (CapabilityName == "AtomicFloat16MinMaxEXT") {
    if (isCoreChildOf(Platform, IGFX_XE3P_CORE)) return true;
  }
  if (CapabilityName == "AtomicFloat32MinMaxEXT") {
    if (true /* Supported on all platforms */) return true;
  }
  if (CapabilityName == "AtomicFloat64MinMaxEXT") {
    if (isProductChildOf(Platform, IGFX_METEORLAKE)) return true;
  }
  if (CapabilityName == "AtomicInt16CompareExchangeINTEL") {
    if (includeExperimental) {
      if (isCoreChildOf(Platform, IGFX_XE3P_CORE)) return true;
    }
  }
  if (CapabilityName == "Int16AtomicsINTEL") {
    if (includeExperimental) {
      if (false /* Not supported */) return true;
    }
  }
  if (CapabilityName == "AtomicBFloat16LoadStoreINTEL") {
    if (includeExperimental) {
      if (isCoreChildOf(Platform, IGFX_XE3P_CORE)) return true;
    }
  }
  if (CapabilityName == "AtomicBFloat16AddINTEL") {
    if (includeExperimental) {
      if (isCoreChildOf(Platform, IGFX_XE3P_CORE)) return true;
    }
  }
  if (CapabilityName == "AtomicBFloat16MinMaxINTEL") {
    if (includeExperimental) {
      if (isCoreChildOf(Platform, IGFX_XE3P_CORE)) return true;
    }
  }
  if (CapabilityName == "Subgroup2DBlockIOINTEL") {
    if (isCoreChildOf(Platform, IGFX_XE_HPC_CORE)) return true;
  }
  if (CapabilityName == "Subgroup2DBlockTransformINTEL") {
    if (isCoreChildOf(Platform, IGFX_XE_HPC_CORE)) return true;
  }
  if (CapabilityName == "Subgroup2DBlockTransposeINTEL") {
    if (isCoreChildOf(Platform, IGFX_XE_HPC_CORE)) return true;
  }
  if (CapabilityName == "ArbitraryPrecisionIntegersINTEL") {
    if (includeExperimental) {
      if (true /* Supported on all platforms */) return true;
    }
  }
  if (CapabilityName == "BFloat16ArithmeticINTEL") {
    if (includeExperimental) {
      if ((isCoreChildOf(Platform, IGFX_XE_HPG_CORE)) && (!(Platform.eProductFamily == IGFX_METEORLAKE))) return true;
    }
  }
  if (CapabilityName == "BFloat16ConversionINTEL") {
    if ((isCoreChildOf(Platform, IGFX_XE_HPG_CORE)) && (!(Platform.eProductFamily == IGFX_METEORLAKE))) return true;
  }
  if (CapabilityName == "BindlessImagesINTEL") {
    if (includeExperimental) {
      if ((isCoreChildOf(Platform, IGFX_XE_HPG_CORE)) && (!(Platform.eProductFamily == IGFX_PVC)) && (!(Platform.eProductFamily == IGFX_CRI))) return true;
    }
  }
  if (CapabilityName == "CacheControlsINTEL") {
    if (true /* Supported on all platforms */) return true;
  }
  if (CapabilityName == "DebugInfoModuleINTEL") {
    if (includeExperimental) {
      if (true /* Supported on all platforms */) return true;
    }
  }
  if (CapabilityName == "Float4TypeINTEL") {
    if (includeExperimental) {
      if (isCoreChildOf(Platform, IGFX_XE3P_CORE)) return true;
    }
  }
  if (CapabilityName == "Float4CooperativeMatrixINTEL") {
    if (includeExperimental) {
      if (isCoreChildOf(Platform, IGFX_XE3P_CORE)) return true;
    }
  }
  if (CapabilityName == "FloatConversionsINTEL") {
    if (includeExperimental) {
      if (isCoreChildOf(Platform, IGFX_XE3P_CORE)) return true;
    }
  }
  if (CapabilityName == "FPFastMathModeINTEL") {
    if (includeExperimental) {
      if (true /* Supported on all platforms */) return true;
    }
  }
  if (CapabilityName == "FPMaxErrorINTEL") {
    if (includeExperimental) {
      if (true /* Supported on all platforms */) return true;
    }
  }
  if (CapabilityName == "FunctionPointersINTEL") {
    if (includeExperimental) {
      if (true /* Supported on all platforms */) return true;
    }
  }
  if (CapabilityName == "IndirectReferencesINTEL") {
    if (includeExperimental) {
      if (true /* Supported on all platforms */) return true;
    }
  }
  if (CapabilityName == "GlobalVariableHostAccessINTEL") {
    if (true /* Supported on all platforms */) return true;
  }
  if (CapabilityName == "AsmINTEL") {
    if (includeExperimental) {
      if (true /* Supported on all platforms */) return true;
    }
  }
  if (CapabilityName == "Int4TypeINTEL") {
    if (includeExperimental) {
      if (isCoreChildOf(Platform, IGFX_XE3P_CORE)) return true;
    }
  }
  if (CapabilityName == "Int4CooperativeMatrixINTEL") {
    if (includeExperimental) {
      if (false /* Not supported */) return true;
    }
  }
  if (CapabilityName == "PackedCooperativeMatrixINTEL") {
    if (includeExperimental) {
      if (isCoreChildOf(Platform, IGFX_XE_HPG_CORE)) return true;
    }
  }
  if (CapabilityName == "CooperativeMatrixInvocationInstructionsINTEL") {
    if (includeExperimental) {
      if (isCoreChildOf(Platform, IGFX_XE_HPG_CORE)) return true;
    }
  }
  if (CapabilityName == "CooperativeMatrixTF32ComponentTypeINTEL") {
    if (includeExperimental) {
      if (isCoreChildOf(Platform, IGFX_XE_HPC_CORE)) return true;
    }
  }
  if (CapabilityName == "CooperativeMatrixBFloat16ComponentTypeINTEL") {
    if (includeExperimental) {
      if (isCoreChildOf(Platform, IGFX_XE_HPG_CORE)) return true;
    }
  }
  if (CapabilityName == "CooperativeMatrixPrefetchINTEL") {
    if (includeExperimental) {
      if (isCoreChildOf(Platform, IGFX_XE_HPC_CORE)) return true;
    }
  }
  if (CapabilityName == "KernelAttributesINTEL") {
    if (includeExperimental) {
      if (true /* Supported on all platforms */) return true;
    }
  }
  if (CapabilityName == "FPGAKernelAttributesINTEL") {
    if (includeExperimental) {
      if (false /* Not supported */) return true;
    }
  }
  if (CapabilityName == "FPGAKernelAttributesv2INTEL") {
    if (includeExperimental) {
      if (false /* Not supported */) return true;
    }
  }
  if (CapabilityName == "CapabilityLongCompositesINTEL") {
    if (true /* Supported on all platforms */) return true;
  }
  if (CapabilityName == "SubgroupImageMediaBlockIOINTEL") {
    if ((Platform.eRenderCoreFamily == IGFX_GEN12LP_CORE) || (Platform.eRenderCoreFamily == IGFX_XE_HPG_CORE)) return true;
  }
  if (CapabilityName == "MemoryAccessAliasingINTEL") {
    if (includeExperimental) {
      if (true /* Supported on all platforms */) return true;
    }
  }
  if (CapabilityName == "OptNoneINTEL") {
    if (includeExperimental) {
      if (true /* Supported on all platforms */) return true;
    }
  }
  if (CapabilityName == "PredicatedIOINTEL") {
    if (includeExperimental) {
      if (true /* Supported on all platforms */) return true;
    }
  }
  if (CapabilityName == "SigmoidINTEL") {
    if (includeExperimental) {
      if (true /* Supported on all platforms */) return true;
    }
  }
  if (CapabilityName == "SplitBarrierINTEL") {
    if (true /* Supported on all platforms */) return true;
  }
  if (CapabilityName == "SubgroupBufferPrefetchINTEL") {
    if (true /* Supported on all platforms */) return true;
  }
  if (CapabilityName == "SubgroupMatrixMultiplyAccumulateINTEL") {
    if ((isCoreChildOf(Platform, IGFX_XE_HPG_CORE)) && (!(Platform.eProductFamily == IGFX_METEORLAKE)) && (!(Platform.eProductFamily == IGFX_ARROWLAKE))) return true;
  }
  if (CapabilityName == "Subgroup​Scaled​Matrix​Multiply​Accumulate​INTEL") {
    if (includeExperimental) {
      if (isCoreChildOf(Platform, IGFX_XE3P_CORE)) return true;
    }
  }
  if (CapabilityName == "SubgroupShuffleINTEL") {
    if (true /* Supported on all platforms */) return true;
  }
  if (CapabilityName == "SubgroupBufferBlockIOINTEL") {
    if (true /* Supported on all platforms */) return true;
  }
  if (CapabilityName == "SubgroupImageBlockIOINTEL") {
    if (true /* Supported on all platforms */) return true;
  }
  if (CapabilityName == "TensorFloat32RoundingINTEL") {
    if (isCoreChildOf(Platform, IGFX_XE_HPC_CORE)) return true;
  }
  if (CapabilityName == "UnstructuredLoopControlsINTEL") {
    if (true /* Supported on all platforms */) return true;
  }
  if (CapabilityName == "VariableLengthArrayINTEL") {
    if (includeExperimental) {
      if (true /* Supported on all platforms */) return true;
    }
  }
  if (CapabilityName == "UntypedVariableLengthArrayINTEL") {
    if (includeExperimental) {
      if (true /* Supported on all platforms */) return true;
    }
  }
  if (CapabilityName == "VectorComputeINTEL") {
    if (includeExperimental) {
      if (true /* Supported on all platforms */) return true;
    }
  }
  if (CapabilityName == "VectorAnyINTEL") {
    if (includeExperimental) {
      if (true /* Supported on all platforms */) return true;
    }
  }
  if (CapabilityName == "BFloat16TypeKHR") {
    if (includeExperimental) {
      if ((isCoreChildOf(Platform, IGFX_XE_HPG_CORE)) && (!(Platform.eProductFamily == IGFX_METEORLAKE))) return true;
    }
  }
  if (CapabilityName == "BFloat16DotProductKHR") {
    if (includeExperimental) {
      if ((isCoreChildOf(Platform, IGFX_XE_HPG_CORE)) && (!(Platform.eProductFamily == IGFX_METEORLAKE))) return true;
    }
  }
  if (CapabilityName == "BFloat16CooperativeMatrixKHR") {
    if (includeExperimental) {
      if ((isCoreChildOf(Platform, IGFX_XE_HPG_CORE)) && (!(Platform.eProductFamily == IGFX_METEORLAKE))) return true;
    }
  }
  if (CapabilityName == "BitInstructions") {
    if (true /* Supported on all platforms */) return true;
  }
  if (CapabilityName == "CooperativeMatrixKHR") {
    if (includeExperimental) {
      if (isCoreChildOf(Platform, IGFX_XE_HPG_CORE)) return true;
    }
  }
  if (CapabilityName == "ExpectAssumeKHR") {
    if (true /* Supported on all platforms */) return true;
  }
  if (CapabilityName == "DotProductKHR") {
    if (true /* Supported on all platforms */) return true;
  }
  if (CapabilityName == "DotProductInputAllKHR") {
    if (true /* Supported on all platforms */) return true;
  }
  if (CapabilityName == "DotProductInput4x8BitKHR") {
    if (true /* Supported on all platforms */) return true;
  }
  if (CapabilityName == "DotProductInput4x8BitPackedKHR") {
    if (true /* Supported on all platforms */) return true;
  }
  if (CapabilityName == "Linkage") {
    if (true /* Supported on all platforms */) return true;
  }
  if (CapabilityName == "ShaderClockKHR") {
    if (true /* Supported on all platforms */) return true;
  }
  if (CapabilityName == "GroupUniformArithmeticKHR") {
    if (includeExperimental) {
      if (true /* Supported on all platforms */) return true;
    }
  }
  return false;
}

// Get extension info with capabilities for a platform
inline std::vector<SPIRVExtension> getSupportedExtensionInfo(PLATFORM Platform, bool includeExperimental = false) {
  std::vector<SPIRVExtension> SupportedExtensions;
  for (const auto& Ext : AllExtensions) {
    if (isExtensionSupported(Ext.Name, Platform, includeExperimental)) {
      SPIRVExtension SupportedExt;
      SupportedExt.Name = Ext.Name;
      SupportedExt.SpecURL = Ext.SpecURL;
      for (const auto& Cap : Ext.Capabilities) {
        if (isCapabilitySupported(Cap.Name, Platform, includeExperimental)) {
          SupportedExt.Capabilities.push_back(Cap);
        }
      }
      SupportedExtensions.push_back(SupportedExt);
    }
  }
  return SupportedExtensions;
}
} // namespace SPIRVExtensionsSupport
} // namespace IGC

#endif // IGCC_SPIRV_EXTENSIONS_SUPPORT_H
