/***************************************************************************
                          kplgraph.h  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Wed Sep 08 1999
    copyright            : (C) 2002 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef KPLGRAPH_H
#define KPLGRAPH_H

class QRect;
class QString;

/**
  * Abstract graphics class. Provides high level plotting functionality.
  * @author Werner Stille
  */
class KplGraph
{
public:
  /** Constructor */
  KplGraph();
  /** Destructor */
  virtual ~KplGraph();
  /** Types of devices */
  enum DevTypes{Screen, Printer, PS};
  /** Grid modes */
  enum GridModes{Nothing = -1, FrameOnly, Axes, AxesWithLabels, Grid,
                 GridWithLabels};
  /** Returns device type
   * @return device type
   */
  DevTypes type() const;
  /**
   * Converts from centimeters to user units.
   * @param xcm x value in centimeters.
   * @param ycm y value in centimeters.
   * @param fxn normalization factor for x values.
   * @param fyn normalization factor for y values.
   * @param xr pointer to x value in user units (output).
   * @param yr pointer to y value in user units (output).
   */
  virtual void cm2r(double xcm, double ycm, double fxn, double fyn,
                    double* xr, double* yr) const = 0;
  /**
   * (Re)sets page size and resets scaling to centimeters.
   * @param x page width in cm.
   * @param y page height in cm.
   */
  virtual void format(double x, double y) = 0;
  /**
   * Sets color for frame and labels.
   * @param icol1 color.
   */
  virtual void setColFrame(unsigned icol1) = 0;
  /**
   * Sets color for grids and tics.
   * @param icol2 color.
   */
  virtual void setColGrid(unsigned icol2) = 0;
  /**
   * Sets color for data arrays and functions.
   * @param icol3 color.
   */
  virtual void setColData(unsigned icol3) = 0;
  /**
   * Sets all colors.
   * @param icol1 color for frame and labels.
   * @param icol2 color for grids and tics.
   * @param icol3 color for data arrays and functions.
   */
  virtual void setCol(unsigned icol1, unsigned icol2, unsigned icol3) = 0;
  /**
   * Sets relative size of text, markers and lines.
   * @param r relative size.
   * @return old value of relative size.
   */
  virtual double setRelSize(double r) = 0;
  /**
   * Sets position and size of the data window.
   * @param x0 left page margin in centimeters.
   * @param xl width in centimeters.
   * @param y0 bottom page margin in centimeters.
   * @param yl height in centimeters.
   */
  virtual void Window(double x0, double xl, double y0, double yl) = 0;
  /**
   * Scales data window to user units.
   * @param xl x value of left data window margin in user units.
   * @param xr x value of right data window margin in user units.
   * @param yu y value of bottom data window margin in user units.
   * @param yo y value of top window data margin in user units.
   * @param logx true for logarithmic x scale.
   * @param logy true for logarithmic y scale.
   */
  virtual void scale(double xl, double xr, double yu, double yo,
                     bool logx = false, bool logy = false) = 0;
  /**
   * Sets number of digits after decimal point for axis labels.
   * @param nx number of digits for x axis.
   * @param ny number of digits for y axis.
   */
  virtual void setDig(int nx, int ny) = 0;
  /** Draws frame around data window. */
  virtual void frame() = 0;
  /**
   * Draws frame and axes of data window.
   * @param xtic distance between major tics of x axis.
   * @param ytic distance between major tics of y axis.
   * @param intx number of intervals between major tics of x axis.
   * @param inty number of intervals between major tics of y axis.
   * @param mode 0 for frame only, 1 for axes with tics,
   * 2 for axes with tics and labels, 3 for grid, 4 for grid with labels.
   * @param xOff offset for first major x tic.
   * @param yOff offset for first major y tic.
   * @param xBottom true for x axis labels on bottom side.
   * @param xTop true for x axis labels on top side.
   * @param yLeft true for y axis labels on left side.
   * @param yRight true for y axis labels on right side.
   */
  virtual void raster(double xtic, double ytic, int intx, int inty,
                      int mode,  double xOff = 0.0, double yOff = 0.0,
                      bool xBottom = true, bool xTop = false,
                      bool yLeft = true, bool yRight = false) = 0;
  /**
   * Sets angle of orientation for text plotting.
   * @param ang angle in degrees.
   */
  virtual void setDir(double ang) = 0;
  /**
   * Plots text.
   * @param str text.
   * @param x x value in cm.
   * @param y y value in cm.
   * @param iorg 1: left aligned, 2: centered, 3: right aligned.
   * @param bRect pointer to bounding rectangle, calculated when non-null.
   */
  virtual void textcm(const QString& str, double x, double y, int iorg,
                      QRect* bRect = 0) = 0;
  /**
   * Plots arrow.
   * @param x x position of head in cm.
   * @param y y position of head in cm.
   * @param dir direction in degree.
   * @param len total length in cm.
   * @param bRect pointer to bounding rectangle, calculated when non-null.
   */
  virtual void arrow(double x, double y, double dir, double len,
                     QRect* bRect = 0) = 0;
  /**
   * Plots arc.
   * @param x x position of left side in cm.
   * @param y y position of bottom side in cm.
   * @param w width in cm.
   * @param h height in cm.
   * @param a start angle in degree.
   * @param alen length in degree.
   * @param dir rotation angle in degree.
   * @param bRect pointer to bounding rectangle, calculated when non-null.
   */
  virtual void arc(double x, double y, double w, double h, double a,
                   double alen, double dir = 0.0, QRect* bRect = 0) = 0;
  /**
   * Plots rectangle.
   * @param x x position of left side in cm.
   * @param y y position of bottom side in cm.
   * @param w width in cm.
   * @param h height in cm.
   * @param filled true for filled with color.
   * @param dir rotation angle in degree.
   * @param bRect pointer to bounding rectangle, calculated when non-null.
   */
  virtual void rectangle(double x, double y, double w, double h,
                         bool filled, double dir = 0.0, QRect* bRect = 0) = 0;
  /**
   * Plots ellipse.
   * @param x x position of center in cm.
   * @param y y position of center in cm.
   * @param w width in cm.
   * @param h height in cm.
   * @param filled true for filled with color.
   * @param dir rotation angle in degree.
   * @param bRect pointer to bounding rectangle, calculated when non-null.
   */
  virtual void ellipse(double x, double y, double w, double h,
                       bool filled, double dir = 0.0, QRect* bRect = 0) = 0;
  /**
   * Plots heading text.
   * @param str text.
   * @param xTop true for x axis labels on top side.
   */
  virtual void letH(const QString& str, bool xTop = false) = 0;
  /**
   * Plots x axis text.
   * @param str text.
   * @param iex logarithm of normalization factor.
   * @param xTop true for x axis labels on top side.
   */
  virtual void letX(const QString& str, int iex, bool xTop = false) = 0;
  /**
   * Plots y axis text.
   * @param str text.
   * @param iex logarithm of normalization factor.
   * @param yRight true for y axis labels on right side.
   */
  virtual void letY(const QString& str, int iex, bool yRight = false) = 0;
  /**
   * Selects marker or line type.
   * @param is 0 for circles, < 0 for different markers,
   * >0 for different line types.
   */
  virtual void setSymbol(int is) = 0;
  /**
   * Draws markers or line given by an array of data points.
   * @param x array of x values in user units.
   * @param y array of y values in user units.
   * @param fx normalization factor for x values.
   * @param fy normalization factor for y values.
   * @param n number of points.
   */
  virtual void plArray(const double* x, const double* y,
                       double fx, double fy, int n, bool clip = true) = 0;
  /**
   * Plots error bars for an array of data points.
   * @param x array of x values in user units.
   * @param y array of y values in user units.
   * @param e array of errors of the y values in user units.
   * @param fx normalization factor for x values.
   * @param fy normalization factor for y values.
   * @param n number of points.
   * @param clip false for non-clipping by frame rectangle.
   */
  virtual void plError(const double* x, const double* y, const double* e,
                       double fx, double fy, int n) = 0;
  /** Resets dash pattern position. */
  virtual void resetDash() = 0;
  /**
   * Prepares number label.
   * @param a value in user units.
   * @param ndig number of digits after decimal point
   * (-1: no display of decimal point).
   * @param label label (output).
   * @return length of label string.
   */
  static int prenum(double a, int ndig, QString& label);
  /**
   * Converts Qt1 QRgb to QRgb.
   * @param u Qt1 rgb value.
   * @return QRgb value for actual Qt version.
   */
  static unsigned rgbQt(unsigned u);
  /**
   * Converts QRgb to Qt1 QRgb.
   * @param rgb QRgb value of actual Qt version.
   * @return QRgb value for actual Qt1 version.
   */
  static unsigned rgbQt1(unsigned rgb);

protected:
  bool logx, logy;
  int kSymb;
  double relSize;
  double fxn, fyn;
  DevTypes devType;

  friend class ScaledItem;
  friend class FunItem;
  friend class SplineItem;
};

#endif
