<?php
/**
 * Class representing vEvents
 *
 * $Horde: horde/lib/iCalendar/vevent.php,v 1.13 2003/07/03 09:16:39 mikec Exp $
 *
 * Copyright 2003 Mike Cochrane <mike@graftonhall.co.nz>
 *
 * See the enclosed file COPYING for license information (LGPL). If you
 * did not receive this file, see http://www.fsf.org/copyleft/lgpl.html.
 *
 * @author  Mike Cochrane <mike@graftonhall.co.nz>
 * @version $Revision: 1.13 $
 * @since   Horde 3.0
 * @package horde.iCalendar
 */
 class Horde_iCalendar_vevent extends Horde_iCalendar {

    function getType()
    {
        return 'vEvent';
    }

    function parsevCalendar($data)
    {
        parent::parsevCalendar($data, 'VEVENT');
    }

    function exportvCalendar(&$container)
    {
        // Default values
        $requiredAttributes = array();
        $requiredAttributes['DTSTAMP']   = time();
        $requiredAttributes['ORGANIZER'] = _('Unknown Organizer');
        $requiredAttributes['UID']       = $this->_exportDateTime(time()) . '@' . $_SERVER["SERVER_NAME"];
        switch ($container->getAttribute('METHOD')) {
            case 'PUBLISH':
                $requiredAttributes['DTSTART']   = time();
                $requiredAttributes['SUMMARY']   = '';
                break;

            case 'REQUEST':
                $requiredAttributes['ATTENDEE']  = '';
                $requiredAttributes['DTSTART']   = time();
                $requiredAttributes['SUMMARY']   = '';
                break;

            case 'REPLY':
                $requiredAttributes['ATTENDEE']  = '';
                break;

            case 'ADD':
                $requiredAttributes['DTSTART']   = time();
                $requiredAttributes['SEQUENCE']  = 1;
                $requiredAttributes['SUMMARY']   = '';
                break;

            case 'CANCEL':
                $requiredAttributes['ATTENDEE']  = '';
                $requiredAttributes['SEQUENCE']  = 1;
                break;

            case 'REFRESH':
                $requiredAttributes['ATTENDEE']  = '';
                break;

        }

        foreach ($requiredAttributes as $name => $default_value) {
            if (is_a($this->getAttribute($name), 'PEAR_Error')) {
                $this->setAttribute($name, $default_value);
            }
        }

        return parent::_exportvData('VEVENT');
    }

    /**
     * Update the status of an attendee of an event
     *
     * @param $email    The email address of the attendee
     * @param $status   The participant status to set.
     * @param $fullname The full name of the participant to set.
     */
    function updateAttendee($email, $status, $fullname = '')
    {
        foreach ($this->_attributes as $key => $attribute) {
            if ($attribute['name'] == 'ATTENDEE' && $attribute['value'] == 'MAILTO:' . $email) {
                $this->_attributes[$key]['params']['PARTSTAT'] = $status;
                if (!empty($fullname)) {
                    $this->_attributes[$key]['params']['CN'] = $fullname;
                }
                unset($this->_attributes[$key]['params']['RSVP']);
                return;
            }
        }
        $params = array('PARTSTAT' => $status);
        if (!empty($fullname)) {
            $params['CN'] = $fullname;
        }
        $this->setAttribute('ATTENDEE', 'MAILTO:' . $email, $params);
    }

    /**
     * Convert this event to a Kronolith Hash
     *
     * @return Array    Array containing the details of the event as used
     *                  by Kronolith
     */
    function toKronolithHash()
    {
        $hash = array();
        $title = $this->getAttribute('SUMMARY');
        if (!is_array($title) && !is_a($title, 'PEAR_Error')) {
            $hash['title'] = $title;
        }
        $desc = $this->getAttribute('DESCRIPTION');
        if (!is_array($desc) && !is_a($desc, 'PEAR_Error')) {
            $hash['description'] = $desc;
        }
        $location = $this->getAttribute('LOCATION');
        if (!is_array($location) && !is_a($location, 'PEAR_Error')) {
            $hash['location'] = $location;
        }

        $start = $this->getAttribute('DTSTART');
        if (!is_array($start) && !is_a($start, 'PEAR_Error')) {
            $hash['start_date'] = date('Y', $start) . '-' . date('m', $start) . '-' . date('j', $start);
            $hash['start_time'] = date('H', $start) . ':' . date('i', $start) . ':' . date('s', $start);
        }

        $end = $this->getAttribute('DTEND');
        if (!is_array($end) && !is_a($end, 'PEAR_Error')) {
            $hash['end_date'] = date('Y', $end) . '-' . date('m', $end) . '-' . date('j', $end);
            $hash['end_time'] = date('H', $end) . ':' . date('i', $end) . ':' . date('s', $end);
        }
        return $hash;
    }

    /**
     * Return the organizer display name or email.
     *
     * @return String   The organizer name to display for this event.
     */
    function organizerName()
    {
        $organizer = $this->getAttribute('ORGANIZER', true);
        if (array_key_exists('CN', $organizer[0])) {
            return $organizer[0]['CN'];
        }

        $organizer = parse_url($this->getAttribute('ORGANIZER'));

        return $organizer['path'];
    }

    /**
     * Update this event with details from another event.
     *
     * @param Horde_iCalendar_vEvent $vevent     The vEvent with latest details
     */
    function updateFromvEvent($vevent)
    {
        $newAttributes = $vevent->getAllAttributes();
        foreach ($newAttributes as $newAttribute) {
            $currentValue = $this->getAttribute($newAttribute['name']);
            if (is_a($currentValue, 'PEAR_error')) {
                // already exists so just add it
                $this->setAttribute($newAttribute['name'], $newAttribute['value'], $newAttribute['params']);
            } else {
                // already exists so locate and modify
                $found = false;
                // Try matching the attribte name and value incase only the
                // params changed (eg attendee updating status).
                foreach ($this->_attributes as $id => $attr) {
                    if ($attr['name'] == $newAttribute['name'] &&
                        $attr['value'] == $newAttribute['value']) {
                        // merge the params
                        foreach ($newAttribute['params'] as $param_id => $param_name) {
                            $this->_attributes[$id]['params'][$param_id] = $param_name;
                        }
                        $found = true;
                        break;
                    }
                }
                if (!$found) {
                    // Else match the first attribute with the same name (eg changing
                    // start time)
                    foreach ($this->_attributes as $id => $attr) {
                        if ($attr['name'] == $newAttribute['name']) {
                            $this->_attributes[$id]['value'] = $newAttribute['value'];
                            // merge the params
                            foreach ($newAttribute['params'] as $param_id => $param_name) {
                                $this->_attributes[$id]['params'][$param_id] = $param_name;
                            }
                            break;
                        }
                    }
                }
            }
        }
    }

    /**
     * Update just the attendess of event with details from another event.
     *
     * @param Horde_iCalendar_vEvent $vevent     The vEvent with latest details
     */
    function updateAttendeesFromvEvent($vevent)
    {
        $newAttributes = $vevent->getAllAttributes();
        foreach ($newAttributes as $newAttribute) {
            if (!$newAttribute['name'] == 'ATTENDEE') {
                continue;
            }
            $currentValue = $this->getAttribute($newAttribute['name']);
            if (is_a($currentValue, 'PEAR_error')) {
                // already exists so just add it
                $this->setAttribute($newAttribute['name'], $newAttribute['value'], $newAttribute['params']);
            } else {
                // already exists so locate and modify
                $found = false;
                // Try matching the attribte name and value incase only the
                // params changed (eg attendee updating status).
                foreach ($this->_attributes as $id => $attr) {
                    if ($attr['name'] == $newAttribute['name'] &&
                        $attr['value'] == $newAttribute['value']) {
                        // merge the params
                        foreach ($newAttribute['params'] as $param_id => $param_name) {
                            $this->_attributes[$id]['params'][$param_id] = $param_name;
                        }
                        $found = true;
                        break;
                    }
                }
                if (!$found) {
                    // Else match the first attribute with the same name (eg changing
                    // start time)
                    foreach ($this->_attributes as $id => $attr) {
                        if ($attr['name'] == $newAttribute['name']) {
                            $this->_attributes[$id]['value'] = $newAttribute['value'];
                            // merge the params
                            foreach ($newAttribute['params'] as $param_id => $param_name) {
                                $this->_attributes[$id]['params'][$param_id] = $param_name;
                            }
                            break;
                        }
                    }
                }
            }
        }
    }

}
