/***************************************************************************
 * SPDX-FileCopyrightText: 2024 S. MANKOWSKI stephane@mankowski.fr
 * SPDX-FileCopyrightText: 2024 G. DE BURE support@mankowski.fr
 * SPDX-License-Identifier: GPL-3.0-or-later
 ***************************************************************************/
/** @file
 * This file is a test script.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgtestmacro.h"
#include "skgbankincludes.h"
#include "skgimportexportmanager.h"
#include <utility>

/**
 * The main function of the unit test
 * @param argc the number of arguments
 * @param argv the list of arguments
 */
int main(int argc, char** argv)
{
    Q_UNUSED(argc)
    Q_UNUSED(argv)

    // Init test
    SKGINITTEST(true)
    QString path = qgetenv("PATH");
    QString in = qgetenv("IN");

    {
        // Test import Weboob
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.load()"), document1.load(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportbackend/fake1/fake1.skg"), true)
        SKGError err;
        {
            SKGTESTBOOL("PUTENV.fake1", qputenv("PATH", (in + "/skgtestimportbackend/fake1/:" + path).toLatin1()), true)

            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_WEBOOB"), err)

            SKGImportExportManager imp1(&document1, QUrl(QLatin1String(".weboob")));
            SKGTESTERROR(QLatin1String("WEBOOB.importFile"), imp1.importFile(), true)
        }

        // Check
        {
            SKGAccountObject account;
            SKGTESTERROR(QLatin1String("ACCOUNT.getObjectByName"), SKGNamedObject::getObjectByName(&document1, QLatin1String("v_account"), QLatin1String("12345"), account), true)
            SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
            SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("1523.99"))
            SKGTEST(QLatin1String("ACCOUNT:getAmount"), SKGServices::doubleToString(account.getAmount(QDate(2013, 5, 1))), QLatin1String("1471.22"))

            SKGTESTERROR(QLatin1String("ACCOUNT.getObjectByName"), SKGNamedObject::getObjectByName(&document1, QLatin1String("v_account"), QLatin1String("47896"), account), true)
            SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
            SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-30"))
            SKGTEST(QLatin1String("ACCOUNT:getAmount"), SKGServices::doubleToString(account.getAmount(QDate(2013, 5, 1))), QLatin1String("0"))

            SKGObjectBase::SKGListSKGObjectBase result;
            SKGTESTERROR(QLatin1String("DOC.getObjects"), document1.getObjects(QLatin1String("v_operation"), QLatin1String("t_comment='SNCF'"), result), true)
            SKGTEST(QLatin1String("DOC.getObjects.count"), result.count(), 2)

            SKGTESTERROR(QLatin1String("DOC.getObjects"), document1.getObjects(QLatin1String("v_operation"), QLatin1String("d_date='2013-05-26'"), result), true)
            SKGTEST(QLatin1String("DOC.getObjects.count"), result.count(), 1)
        }
    }

    {
        // BUG: 320716
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            SKGTESTBOOL("PUTENV.320716_1", qputenv("PATH", (in + "/skgtestimportbackend/320716_1/:" + path).toLatin1()), true)

            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_WEBOOB"), err)

            SKGImportExportManager imp1(&document1, QUrl(QLatin1String(".weboob")));
            SKGTESTERROR(QLatin1String("WEBOOB.importFile"), imp1.importFile(), true)
        }

        // Check
        {
            SKGAccountObject account;
            SKGTESTERROR(QLatin1String("ACCOUNT.getObjectByName"), SKGNamedObject::getObjectByName(&document1, QLatin1String("v_account"), QLatin1String("12345"), account), true)
            SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
            SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("100"))
        }

        {
            SKGTESTBOOL("PUTENV.320716_1", qputenv("PATH", (in + "/skgtestimportbackend/320716_2/:" + path).toLatin1()), true)

            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_WEBOOB"), err)

            SKGImportExportManager imp1(&document1, QUrl(QLatin1String(".weboob")));
            SKGTESTERROR(QLatin1String("WEBOOB.importFile"), imp1.importFile(), true)
        }

        // Check
        {
            SKGAccountObject account;
            SKGTESTERROR(QLatin1String("ACCOUNT.getObjectByName"), SKGNamedObject::getObjectByName(&document1, QLatin1String("v_account"), QLatin1String("12345"), account), true)
            SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
            SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("0"))
        }
    }

    {
        // Test error
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            SKGTESTBOOL("PUTENV.fake1", qputenv("PATH", (in + "/skgtestimportbackend/error1/:" + path).toLatin1()), true)

            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_WEBOOB"), err)

            SKGImportExportManager imp1(&document1, QUrl(QLatin1String(".weboob")));
            SKGTEST(QLatin1String("WEBOOB.importFile"), imp1.importFile().getReturnCode(), ERR_FAIL)
        }
    }

    {
        // Test error
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            SKGTESTBOOL("PUTENV.fake1", qputenv("PATH", (in + "/skgtestimportbackend/error2/:" + path).toLatin1()), true)

            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_WEBOOB"), err)

            SKGImportExportManager imp1(&document1, QUrl(QLatin1String(".weboob")));
            SKGTEST(QLatin1String("WEBOOB.importFile"), imp1.importFile().getReturnCode(), ERR_FAIL)
        }
    }

    {
        // Test error
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            SKGTESTBOOL("PUTENV.fake1", qputenv("PATH", (in + "/skgtestimportbackend/error3/:" + path).toLatin1()), true)

            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_WEBOOB"), err)

            document1.sendMessage(QLatin1String("Hello"));

            SKGImportExportManager imp1(&document1, QUrl(QLatin1String(".weboob")));
            SKGTEST(QLatin1String("WEBOOB.importFile"), imp1.importFile().getReturnCode(), ERR_FAIL)
        }
    }

    {
        // Test error
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            SKGTESTBOOL("PUTENV.fake1", qputenv("PATH", (in + "/skgtestimportbackend/error4/:" + path).toLatin1()), true)

            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_WEBOOB"), err)

            SKGImportExportManager imp1(&document1, QUrl(QLatin1String(".weboob")));
            SKGTEST(QLatin1String("WEBOOB.importFile"), imp1.importFile().getReturnCode(), ERR_FAIL)
        }
    }

    {
        // Better account selection
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("CREATE ACCOUNT"), err)

            // Creation the bank
            SKGBankObject bank(&document1);
            SKGTESTERROR(QLatin1String("BANK.setName"), bank.setName(QLatin1String("caisse-epargne")), true)
            SKGTESTERROR(QLatin1String("BANK.setNumber"), bank.setNumber(QLatin1String("13135")), true)
            SKGTESTERROR(QLatin1String("BANK.save"), bank.save(), true)

            // Creation the account
            SKGAccountObject account;
            SKGTESTERROR(QLatin1String("BANK.addAccount"), bank.addAccount(account), true)
            SKGTESTERROR(QLatin1String("ACCOUNT.setName"), account.setName(QLatin1String("test")), true)
            SKGTESTERROR(QLatin1String("ACCOUNT.setNumber"), account.setNumber(QLatin1String("123456789")), true)
            SKGTESTERROR(QLatin1String("ACCOUNT.setAgencyNumber"), account.setAgencyNumber(QLatin1String("00080")), true)
            SKGTESTERROR(QLatin1String("ACCOUNT.save"), account.save(), true)
        }
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_WEBOOB"), err)

            SKGTESTBOOL("PUTENV.weboob", qputenv("PATH", (in + "/skgtestimportbackend/betterselection/:" + path).toLatin1()), true)

            SKGImportExportManager imp1(&document1, QUrl(QLatin1String(".weboob")));
            SKGTESTERROR(QLatin1String("WEBOOB.importFile"), imp1.importFile(), true)
        }

        // Check
        {
            SKGAccountObject account;
            SKGTESTERROR(QLatin1String("ACCOUNT.getObjectByName"), SKGNamedObject::getObjectByName(&document1, QLatin1String("v_account"), QLatin1String("test"), account), true)
            SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
            SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-147994.82"))
        }
    }

    {
        // Better account selection
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_WEBOOB"), err)

            SKGTESTBOOL("PUTENV.weboob", qputenv("PATH", (in + "/skgtestimportbackend/joint_accounts/:" + path).toLatin1()), true)

            SKGImportExportManager imp1(&document1, QUrl(QLatin1String(".weboob")));
            SKGTESTERROR(QLatin1String("WEBOOB.importFile"), imp1.importFile(), true)
        }

        // Check
        {
            SKGAccountObject account;
            SKGTESTERROR(QLatin1String("ACCOUNT.getObjectByName"), SKGNamedObject::getObjectByName(&document1, QLatin1String("v_account"), QLatin1String("12345"), account), true)
            SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
            SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("1000"))
        }
    }

    {
        // BULK import
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_BULK"), err)

            SKGTESTBOOL("PUTENV.bulk", qputenv("PATH", (in + "/skgtestimportbackend/bulk/:" + path).toLatin1()), true)

            SKGImportExportManager imp1(&document1, QUrl(QLatin1String(".bulk")));
            SKGTESTERROR(QLatin1String("WEBOOB.importFile"), imp1.importFile(), true)
        }

        // Check
        {
            SKGAccountObject account;
            SKGTESTERROR(QLatin1String("ACCOUNT.getObjectByName"), SKGNamedObject::getObjectByName(&document1, QLatin1String("v_account"), QLatin1String("12345"), account), true)
            SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
            SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("10"))
        }

        // Check
        {
            SKGAccountObject account;
            SKGTESTERROR(QLatin1String("ACCOUNT.getObjectByName"), SKGNamedObject::getObjectByName(&document1, QLatin1String("v_account"), QLatin1String("78900"), account), true)
            SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
            SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("20"))
        }
    }

    {
        // STAMM
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("CREATE ACCOUNT"), err)

            // Creation the bank
            SKGBankObject bank(&document1);
            SKGTESTERROR(QLatin1String("BANK.setName"), bank.setName(QLatin1String("ldlc")), true)
            SKGTESTERROR(QLatin1String("BANK.save"), bank.save(), true)

            // Creation the accounts
            {
                SKGAccountObject account;
                SKGTESTERROR(QLatin1String("BANK.addAccount"), bank.addAccount(account), true)
                SKGTESTERROR(QLatin1String("ACCOUNT.setName"), account.setName(QLatin1String("acc1")), true)
                SKGTESTERROR(QLatin1String("ACCOUNT.setNumber"), account.setNumber(QLatin1String("1234567A")), true)
                SKGTESTERROR(QLatin1String("ACCOUNT.setAgencyNumber"), account.setAgencyNumber(QLatin1String("00080")), true)
                SKGTESTERROR(QLatin1String("ACCOUNT.save"), account.save(), true)
            }
            {
                SKGAccountObject account;
                SKGTESTERROR(QLatin1String("BANK.addAccount"), bank.addAccount(account), true)
                SKGTESTERROR(QLatin1String("ACCOUNT.setName"), account.setName(QLatin1String("acc2")), true)
                SKGTESTERROR(QLatin1String("ACCOUNT.setNumber"), account.setNumber(QLatin1String("987654321")), true)
                SKGTESTERROR(QLatin1String("ACCOUNT.setAgencyNumber"), account.setAgencyNumber(QLatin1String("00080")), true)
                SKGTESTERROR(QLatin1String("ACCOUNT.save"), account.save(), true)
            }
            {
                SKGAccountObject account;
                SKGTESTERROR(QLatin1String("BANK.addAccount"), bank.addAccount(account), true)
                SKGTESTERROR(QLatin1String("ACCOUNT.setName"), account.setName(QLatin1String("acc3")), true)
                SKGTESTERROR(QLatin1String("ACCOUNT.setNumber"), account.setNumber(QLatin1String("1111111A")), true)
                SKGTESTERROR(QLatin1String("ACCOUNT.setAgencyNumber"), account.setAgencyNumber(QLatin1String("00080")), true)
                SKGTESTERROR(QLatin1String("ACCOUNT.save"), account.save(), true)
            }
        }
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_WEBOOB"), err)

            SKGTESTBOOL("PUTENV.weboob", qputenv("PATH", (in + "/skgtestimportbackend/stamm/:" + path).toLatin1()), true)

            SKGImportExportManager imp1(&document1, QUrl(QLatin1String(".weboob")));
            SKGTESTERROR(QLatin1String("WEBOOB.importFile"), imp1.importFile(), true)
        }

        // Checks
        {
            SKGAccountObject account;
            SKGTESTERROR(QLatin1String("ACCOUNT.getObjectByName"), SKGNamedObject::getObjectByName(&document1, QLatin1String("v_account"), QLatin1String("acc1"), account), true)
            SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
            SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-50"))
        }
        {
            SKGAccountObject account;
            SKGTESTERROR(QLatin1String("ACCOUNT.getObjectByName"), SKGNamedObject::getObjectByName(&document1, QLatin1String("v_account"), QLatin1String("acc2"), account), true)
            SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
            SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-50"))
        }
        {
            SKGAccountObject account;
            SKGTESTERROR(QLatin1String("ACCOUNT.getObjectByName"), SKGNamedObject::getObjectByName(&document1, QLatin1String("v_account"), QLatin1String("acc3"), account), true)
            SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
            SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-50"))
        }

        {
            SKGAccountObject account;
            SKGTESTERROR(QLatin1String("ACCOUNT.getObjectByName"), SKGNamedObject::getObjectByName(&document1, QLatin1String("v_account"), QLatin1String("1234567W089"), account), true)
            SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
            SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-50"))
        }

        {
            SKGAccountObject account;
            SKGTESTERROR(QLatin1String("ACCOUNT.getObjectByName"), SKGNamedObject::getObjectByName(&document1, QLatin1String("v_account"), QLatin1String("MYACCOUNTNAME"), account), true)
            SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
            SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-50"))
        }
    }


    {
        // Avoid merge of account due to same name
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_WEBOOB"), err)

            SKGTESTBOOL("PUTENV.weboob", qputenv("PATH", (in + "/skgtestimportbackend/merge/:" + path).toLatin1()), true)

            SKGImportExportManager imp1(&document1, QUrl(QLatin1String(".weboob")));
            SKGTESTERROR(QLatin1String("WEBOOB.importFile"), imp1.importFile(), true)
        }

        // Checks
        {
            SKGAccountObject account;
            SKGTESTERROR(QLatin1String("ACCOUNT.getObjectByName"), SKGNamedObject::getObjectByName(&document1, QLatin1String("v_account"), QLatin1String("CPT"), account), true)
            SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
            SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-50"))
        }
        {
            SKGAccountObject account;
            SKGTESTERROR(QLatin1String("ACCOUNT.getObjectByName"), SKGNamedObject::getObjectByName(&document1, QLatin1String("v_account"), QLatin1String("CPT2"), account), true)
            SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
            SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-50"))
        }
    }

    {
        // Avoid merge of account due to same name
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_WEBOOB"), err)

            SKGTESTBOOL("PUTENV.weboob", qputenv("PATH", (in + "/skgtestimportbackend/kevin/:" + path).toLatin1()), true)

            SKGImportExportManager imp1(&document1, QUrl(QLatin1String(".weboob")));
            SKGTESTERROR(QLatin1String("WEBOOB.importFile"), imp1.importFile(), true)
        }

        // Checks
        QStringList accounts;
        accounts << QLatin1String("COMPTE 1") << QLatin1String("COMPTE 2") << QLatin1String("LIVRET 1") << QLatin1String("LIVRET 2") << QLatin1String("EPARGNE 1") << QLatin1String("EPARGNE 2") << QLatin1String("EPARGNE 3");
        for (const auto& acc : std::as_const(accounts)) {
            SKGAccountObject account;
            SKGTESTERROR(QLatin1String("ACCOUNT.getObjectByName"), SKGNamedObject::getObjectByName(&document1, QLatin1String("v_account"), acc, account), true)
            SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
            SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-10"))
        }
    }

    {
        // Avoid merge of account due to same name
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_WEBOOB"), err)

            SKGTESTBOOL("PUTENV.weboob", qputenv("PATH", (in + "/skgtestimportbackend/double/:" + path).toLatin1()), true)

            SKGImportExportManager imp1(&document1, QUrl(QLatin1String(".weboob")));
            SKGTESTERROR(QLatin1String("WEBOOB.importFile"), imp1.importFile(), true)
        }

        // Checks
        {
            SKGAccountObject account;
            SKGTESTERROR(QLatin1String("ACCOUNT.getObjectByName"), SKGNamedObject::getObjectByName(&document1, QLatin1String("v_account"), QLatin1String("CPT"), account), true)
            SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
            SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-100"))
        }
    }

    {
        // Avoid merge of account due to same name
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_WEBOOB"), err)

            SKGTESTBOOL("PUTENV.weboob", qputenv("PATH", (in + "/skgtestimportbackend/397055/:" + path).toLatin1()), true)

            SKGImportExportManager imp1(&document1, QUrl(QLatin1String(".weboob")));
            SKGTESTERROR(QLatin1String("WEBOOB.importFile"), imp1.importFile(), true)
        }

        // Checks
        {
            SKGAccountObject account;
            SKGTESTERROR(QLatin1String("ACCOUNT.getObjectByName"), SKGNamedObject::getObjectByName(&document1, QLatin1String("v_account"), QLatin1String("LEO"), account), true)
            SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
            SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-20"))
        }
    }

    {
        // rdate Not available
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_WEBOOB"), err)

            SKGTESTBOOL("PUTENV.weboob", qputenv("PATH", (in + "/skgtestimportbackend/397611/:" + path).toLatin1()), true)

            SKGImportExportManager imp1(&document1, QUrl(QLatin1String(".weboob")));
            SKGTESTERROR(QLatin1String("WEBOOB.importFile"), imp1.importFile(), true)
        }

        // Checks
        {
            SKGAccountObject account;
            SKGTESTERROR(QLatin1String("ACCOUNT.getObjectByName"), SKGNamedObject::getObjectByName(&document1, QLatin1String("v_account"), QLatin1String("LEO"), account), true)
            SKGTESTERROR(QLatin1String("ACCOUNT.load"), account.load(), true)
            SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("598.8"))

            SKGObjectBase::SKGListSKGObjectBase result;
            SKGTESTERROR(QLatin1String("DOC.getObjects"), document1.getObjects(QLatin1String("v_operation"), QLatin1String("d_date!='0000-00-00'"), result), true)
            SKGTEST(QLatin1String("DOC.getObjects.count"), result.count(), 3)
        }
    }

    {
        // Double transferts
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_WEBOOB"), err)

            SKGTESTBOOL("PUTENV.weboob", qputenv("PATH", (in + "/skgtestimportbackend/double_transferts/:" + path).toLatin1()), true)

            SKGImportExportManager imp1(&document1, QUrl(QLatin1String(".weboob")));
            SKGTESTERROR(QLatin1String("WEBOOB.importFile"), imp1.importFile(), true)
        }

        // Checks
        {
            SKGObjectBase::SKGListSKGObjectBase result;
            SKGTESTERROR(QLatin1String("DOC.getObjects"), document1.getObjects(QLatin1String("v_operation"), QLatin1String("d_date!='0000-00-00'"), result), true)
            SKGTEST(QLatin1String("DOC.getObjects.count"), result.count(), 4)
        }
    }
    // End test
    SKGENDTEST()
}
