/***************************************************************************
 * SPDX-FileCopyrightText: 2024 S. MANKOWSKI stephane@mankowski.fr
 * SPDX-FileCopyrightText: 2024 G. DE BURE support@mankowski.fr
 * SPDX-License-Identifier: GPL-3.0-or-later
 ***************************************************************************/
/** @file
 * This file is a test script.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */

#include <qdesktopservices.h>

#include "skgbankincludes.h"
#include "skgtestmacro.h"
#include <utility>

/**
 * The main function of the unit test
 * @param argc the number of arguments
 * @param argv the list of arguments
 */
int main(int argc, char** argv)
{
    Q_UNUSED(argc)
    Q_UNUSED(argv)

    // Init test
    SKGINITTEST(true)

    {
        QStringList sources = SKGUnitObject::downloadSources();
        SKGTESTBOOL("UNIT:downloadSources", (sources.count() >= 4), true)
        for (const QString& source : std::as_const(sources)) {
            SKGUnitObject::getCommentFromSource(source);
        }

        // Test unit et unitvalue
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        SKGTEST(QLatin1String("UNIT:getNbTransaction"), document1.getNbTransaction(), 0)
        // Scope of the transaction
        SKGBEGINTRANSACTION(document1, QLatin1String("BANK_T1"), err)

        // Create unit
        SKGUnitObject unit1(&document1);
        SKGTESTERROR(QLatin1String("UNIT:setName"), unit1.setName(QLatin1String("UNIT1")), true)
        SKGTESTERROR(QLatin1String("UNIT:setSymbol"), unit1.setSymbol(QLatin1String("F")), true)
        SKGTEST(QLatin1String("UNIT:getSymbol"), unit1.getSymbol(), QLatin1String("F"))
        SKGTESTERROR(QLatin1String("UNIT:setCountry"), unit1.setCountry(QLatin1String("France")), true)
        SKGTEST(QLatin1String("UNIT:getSymbol"), unit1.getCountry(), QLatin1String("France"))
        SKGTESTERROR(QLatin1String("UNIT:setInternetCode"), unit1.setInternetCode(QLatin1String("code Internet")), true)
        SKGTEST(QLatin1String("UNIT:getInternetCode"), unit1.getInternetCode(), QLatin1String("code Internet"))
        SKGTESTERROR(QLatin1String("UNIT:setType"), unit1.setType(SKGUnitObject::OBJECT), true)
        SKGTEST(QLatin1String("UNIT:getType"), static_cast<unsigned int>(unit1.getType()), static_cast<unsigned int>(SKGUnitObject::OBJECT))
        SKGTESTERROR(QLatin1String("UNIT:setNumberDecimal"), unit1.setNumberDecimal(4), true)
        SKGTEST(QLatin1String("UNIT:getNumberDecimal"), unit1.getNumberDecimal(), 4)

        SKGUnitValueObject unitvalue1;
        SKGTESTERROR(QLatin1String("UNIT:addUnitValue"), unit1.addUnitValue(unitvalue1), false)

        SKGTESTERROR(QLatin1String("UNIT:save"), unit1.save(), true)

        // Create unit value
        SKGTESTERROR(QLatin1String("UNIT:addUnitValue"), unit1.addUnitValue(unitvalue1), true)

        SKGTESTERROR(QLatin1String("UNITVALUE:setQuantity"), unitvalue1.setQuantity(119999.11), true)
        SKGTESTERROR(QLatin1String("UNITVALUE:setDate"), unitvalue1.setDate(SKGServices::stringToDate(QLatin1String("1970-07-16"))), true)
        SKGTESTERROR(QLatin1String("UNITVALUE:save"), unitvalue1.save(), true)

        SKGUnitValueObject unitvalue1bis = SKGUnitValueObject(unitvalue1);
        SKGUnitValueObject unitvalue1ter = SKGUnitValueObject(static_cast<SKGObjectBase>(unitvalue1));
        SKGUnitValueObject unitvalue14(SKGNamedObject(&document1, QLatin1String("xxx"), unitvalue1.getID()));

        SKGUnitObject unit_get;
        SKGTESTERROR(QLatin1String("UNITVALUE:getUnit"), unitvalue1.getUnit(unit_get), true)
        SKGTESTBOOL("UNIT:comparison", (unit_get == unit1), true)

        SKGUnitValueObject unitvalue2;
        SKGTESTERROR(QLatin1String("UNIT:addUnitValue"), unit1.addUnitValue(unitvalue2), true)

        SKGTESTERROR(QLatin1String("UNITVALUE:setQuantity"), unitvalue2.setQuantity(34.12), true)
        SKGTESTERROR(QLatin1String("UNITVALUE:setDate"), unitvalue2.setDate(SKGServices::stringToDate(QLatin1String("1973-02-04"))), true)
        SKGTESTERROR(QLatin1String("UNITVALUE:save"), unitvalue2.save(), true)

        // Retrieve unit
        SKGNamedObject unit2Named;
        SKGTESTERROR(QLatin1String("UNIT:getObjectByName"), SKGNamedObject::getObjectByName(&document1, QLatin1String("unit"), QLatin1String("UNIT1"), unit2Named), true)
        SKGUnitObject unit2(unit2Named);

        // Retrieve unit value
        SKGUnitValueObject unitvalue3;
        SKGTESTERROR(QLatin1String("UNIT:getLastUnitValue"), unit2.getLastUnitValue(unitvalue3), true)

        SKGUnitValueObject unitvalue19700101;
        SKGTESTERROR(QLatin1String("UNIT:getUnitValue"), unit2.getUnitValue(SKGServices::stringToDate(QLatin1String("1970-01-01")), unitvalue19700101), true)
        SKGUnitValueObject unitvalue19700716;
        SKGTESTERROR(QLatin1String("UNIT:getUnitValue"), unit2.getUnitValue(SKGServices::stringToDate(QLatin1String("1970-07-16")), unitvalue19700716), true)
        SKGUnitValueObject unitvalue19720716;
        SKGTESTERROR(QLatin1String("UNIT:getUnitValue"), unit2.getUnitValue(SKGServices::stringToDate(QLatin1String("1972-07-16")), unitvalue19720716), true)
        SKGUnitValueObject unitvalue19730204;
        SKGTESTERROR(QLatin1String("UNIT:getUnitValue"), unit2.getUnitValue(SKGServices::stringToDate(QLatin1String("1973-02-04")), unitvalue19730204), true)
        SKGUnitValueObject unitvalue19750204;
        SKGTESTERROR(QLatin1String("UNIT:getUnitValue"), unit2.getUnitValue(SKGServices::stringToDate(QLatin1String("1975-02-04")), unitvalue19750204), true)

        // Check unit value
        SKGTESTBOOL("UNITVALUE:==", (unitvalue3 == unitvalue2), true)
        SKGTESTBOOL("UNITVALUE:==", (unitvalue19700101 == unitvalue1), true)
        SKGTESTBOOL("UNITVALUE:==", (unitvalue19700716 == unitvalue1), true)
        SKGTESTBOOL("UNITVALUE:==", (unitvalue19720716 == unitvalue1), true)
        SKGTESTBOOL("UNITVALUE:==", (unitvalue19730204 == unitvalue2), true)
        SKGTESTBOOL("UNITVALUE:==", (unitvalue19750204 == unitvalue2), true)

        SKGTEST(QLatin1String("UNITVALUE:getQuantity"), unitvalue1.getQuantity() - 119999.11, 0)
        SKGTEST(QLatin1String("UNITVALUE:getQuantity"), unitvalue3.getQuantity() - 34.12, 0)
        SKGTEST(QLatin1String("UNITVALUE:getDate"), SKGServices::dateToSqlString(unitvalue3.getDate()), QLatin1String("1973-02-04"))

        // Check cascading delete
        SKGTESTERROR(QLatin1String("UNIT:delete"), unit2.remove(), true)

        QStringList oResult;
        SKGTESTERROR(QLatin1String("UNIT:getDistinctValues"), document1.getDistinctValues(QLatin1String("unit"), QLatin1String("id"), oResult), true)
        SKGTEST(QLatin1String("UNIT:oResult.size"), oResult.size(), 0)

        SKGTESTERROR(QLatin1String("UNITVALUE:getDistinctValues"), document1.getDistinctValues(QLatin1String("unitvalue"), QLatin1String("rd_unit_id"), oResult), true)
        SKGTEST(QLatin1String("UNITVALUE:oResult.size"), oResult.size(), 0)

        // Merge
        SKGTESTERROR(QLatin1String("UNIT:merge"), unit1.merge(unit2), true)
    }

    {
        // Test unit et unitvalue
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        SKGTEST(QLatin1String("DOC:getPrimaryUnit"), document1.getPrimaryUnit().Symbol, QLatin1String(""))
        SKGTEST(QLatin1String("DOC:getSecondaryUnit"), document1.getSecondaryUnit().Symbol, QLatin1String(""))
        SKGUnitObject franc(&document1);
        {
            SKGBEGINTRANSACTION(document1, QLatin1String("UNIT_1"), err)

            // Create unit
            SKGTESTERROR(QLatin1String("FRANC:setName"), franc.setName(QLatin1String("F")), true)
            SKGTESTERROR(QLatin1String("FRANC:setSymbol"), franc.setSymbol(QLatin1String("F")), true)
            SKGTESTERROR(QLatin1String("FRANC:setType"), franc.setType(SKGUnitObject::PRIMARY), true)
            SKGTEST(QLatin1String("FRANC:getType"), static_cast<unsigned int>(franc.getType()), static_cast<unsigned int>(SKGUnitObject::PRIMARY))
            SKGTESTERROR(QLatin1String("FRANC:save"), franc.save(), true)

            SKGTEST(QLatin1String("DOC:getPrimaryUnit"), document1.getPrimaryUnit().Symbol, QLatin1String("F"))
        }

        SKGTEST(QLatin1String("DOC:getPrimaryUnit"), document1.getPrimaryUnit().Symbol, QLatin1String("F"))
        SKGTEST(QLatin1String("DOC:getSecondaryUnit"), document1.getSecondaryUnit().Symbol, QLatin1String(""))
        SKGTEST(QLatin1String("DOC:getSecondaryUnit"), document1.getSecondaryUnit().Value, 1)

        {
            SKGBEGINTRANSACTION(document1, QLatin1String("UNIT_2"), err)

            SKGUnitObject euro(&document1);
            SKGTESTERROR(QLatin1String("EURO:setName"), euro.setName(QLatin1String("E")), true)
            SKGTESTERROR(QLatin1String("EURO:setSymbol"), euro.setSymbol(QLatin1String("E")), true)
            SKGTESTERROR(QLatin1String("EURO:setType"), euro.setType(SKGUnitObject::PRIMARY), true)
            SKGTEST(QLatin1String("EURO:getType"), static_cast<unsigned int>(euro.getType()), static_cast<unsigned int>(SKGUnitObject::PRIMARY))
            SKGTESTERROR(QLatin1String("EURO:save"), euro.save(), true)

            SKGTESTERROR(QLatin1String("FRANC:load"), franc.load(), true)
            SKGTEST(QLatin1String("FRANC:getType"), static_cast<unsigned int>(franc.getType()), static_cast<unsigned int>(SKGUnitObject::SECONDARY))

            SKGTEST(QLatin1String("EURO:getType"), static_cast<unsigned int>(euro.getType()), static_cast<unsigned int>(SKGUnitObject::PRIMARY))
            SKGTESTERROR(QLatin1String("EURO:save"), euro.save(), true)
        }

        SKGTEST(QLatin1String("DOC:getPrimaryUnit"), document1.getPrimaryUnit().Symbol, QLatin1String("E"))
        SKGTEST(QLatin1String("DOC:getSecondaryUnit"), document1.getSecondaryUnit().Symbol, QLatin1String("F"))
    }

    {
        // Test unit et unitvalue
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            QStringList units = SKGUnitObject::getListofKnownCurrencies(false);
            SKGTEST(QLatin1String("UNIT:getListofKnownCurrencies"), units.count(), 276)
            units = SKGUnitObject::getListofKnownCurrencies(true);
            SKGTEST(QLatin1String("UNIT:getListofKnownCurrencies"), units.count(), 316)
            for (const auto& unit : std::as_const(units)) {
                SKGBEGINTRANSACTION(document1, QLatin1String("UNIT_1"), err)
                SKGUnitObject unitObj;
                SKGTESTERROR(QLatin1String("UNIT:createCurrencyUnit-") % unit, SKGUnitObject::createCurrencyUnit(&document1, unit, unitObj), true)
                unitObj.getUnitInfo();
            }
            SKGBEGINTRANSACTION(document1, QLatin1String("UNIT_1"), err)
            SKGUnitObject unitObj;
            SKGTESTERROR(QLatin1String("UNIT:createCurrencyUnit"), SKGUnitObject::createCurrencyUnit(&document1, QLatin1String("ZZZZZZZ"), unitObj), false)
        }
    }

    {
        // Compute unit amount at a date
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        SKGUnitObject unit1(&document1);
        SKGUnitObject unit2(&document1);
        {
            SKGBEGINTRANSACTION(document1, QLatin1String("UNIT"), err)
            SKGTESTERROR(QLatin1String("UNIT:setName"), unit1.setName(QLatin1String("U1")), true)
            SKGTESTERROR(QLatin1String("UNIT:setSymbol"), unit1.setSymbol(QLatin1String("U1")), true)
            SKGTESTERROR(QLatin1String("UNIT:save"), unit1.save(), true)
            SKGTESTERROR(QLatin1String("UNIT:addOrModifyUnitValue"), document1.addOrModifyUnitValue(QLatin1String("U1"), SKGServices::stringToDate(QLatin1String("2001-01-01")), 1), true)
            SKGTESTERROR(QLatin1String("UNIT:addOrModifyUnitValue"), document1.addOrModifyUnitValue(QLatin1String("U1"), SKGServices::stringToDate(QLatin1String("2006-01-01")), 6), true)
            SKGTESTERROR(QLatin1String("UNIT:addOrModifyUnitValue"), document1.addOrModifyUnitValue(QLatin1String("U1"), SKGServices::stringToDate(QLatin1String("2007-01-01")), 7), true)
            SKGTESTERROR(QLatin1String("UNIT:addOrModifyUnitValue"), document1.addOrModifyUnitValue(QLatin1String("U1"), SKGServices::stringToDate(QLatin1String("2008-01-01")), 8), true)

            SKGTESTERROR(QLatin1String("UNIT:setName"), unit2.setName(QLatin1String("U2")), true)
            SKGTESTERROR(QLatin1String("UNIT:setSymbol"), unit2.setSymbol(QLatin1String("U2")), true)
            SKGTESTERROR(QLatin1String("UNIT:removeUnit"), unit2.removeUnit(), true)
            SKGTESTERROR(QLatin1String("UNIT:save"), unit2.save(), true)
            SKGTESTERROR(QLatin1String("UNIT:setUnit"), unit2.setUnit(unit2), false)
            SKGTESTERROR(QLatin1String("UNIT:setUnit"), unit2.setUnit(unit1), true)
            SKGTESTERROR(QLatin1String("UNIT:save"), unit2.save(), true)
            SKGTESTERROR(QLatin1String("UNIT:addOrModifyUnitValue"), document1.addOrModifyUnitValue(QLatin1String("U2"), SKGServices::stringToDate(QLatin1String("2001-02-01")), 10), true)
            SKGTESTERROR(QLatin1String("UNIT:addOrModifyUnitValue"), document1.addOrModifyUnitValue(QLatin1String("U2"), SKGServices::stringToDate(QLatin1String("2006-01-01")), 60), true)
            SKGTESTERROR(QLatin1String("UNIT:addOrModifyUnitValue"), document1.addOrModifyUnitValue(QLatin1String("U2"), SKGServices::stringToDate(QLatin1String("2007-02-01")), 70), true)
            SKGTESTERROR(QLatin1String("UNIT:addOrModifyUnitValue"), document1.addOrModifyUnitValue(QLatin1String("U2"), SKGServices::stringToDate(QLatin1String("2008-01-01")), 80), true)
        }

        SKGTEST(QLatin1String("DOC:getAmount"), unit1.getAmount(SKGServices::stringToDate(QLatin1String("2001-01-01"))), 1)
        SKGTEST(QLatin1String("DOC:getAmount"), unit1.getAmount(SKGServices::stringToDate(QLatin1String("2001-02-01"))), 1)
        SKGTEST(QLatin1String("DOC:getAmount"), unit1.getAmount(SKGServices::stringToDate(QLatin1String("2006-01-01"))), 6)
        SKGTEST(QLatin1String("DOC:getAmount"), unit1.getAmount(SKGServices::stringToDate(QLatin1String("2006-02-01"))), 6)
        SKGTEST(QLatin1String("DOC:getAmount"), unit1.getAmount(SKGServices::stringToDate(QLatin1String("2007-01-01"))), 7)
        SKGTEST(QLatin1String("DOC:getAmount"), unit1.getAmount(SKGServices::stringToDate(QLatin1String("2007-02-01"))), 7)
        SKGTEST(QLatin1String("DOC:getAmount"), unit1.getAmount(SKGServices::stringToDate(QLatin1String("2008-01-01"))), 8)
        SKGTEST(QLatin1String("DOC:getAmount"), unit1.getAmount(SKGServices::stringToDate(QLatin1String("2008-02-01"))), 8)

        SKGTEST(QLatin1String("DOC:getAmount"), unit2.getAmount(SKGServices::stringToDate(QLatin1String("2001-01-01"))), 1 * 10)
        SKGTEST(QLatin1String("DOC:getAmount"), unit2.getAmount(SKGServices::stringToDate(QLatin1String("2001-02-01"))), 1 * 10)
        SKGTEST(QLatin1String("DOC:getAmount"), unit2.getAmount(SKGServices::stringToDate(QLatin1String("2006-01-01"))), 6 * 60)
        SKGTEST(QLatin1String("DOC:getAmount"), unit2.getAmount(SKGServices::stringToDate(QLatin1String("2006-02-01"))), 6 * 60)
        SKGTEST(QLatin1String("DOC:getAmount"), unit2.getAmount(SKGServices::stringToDate(QLatin1String("2007-01-01"))), 7 * 60)
        SKGTEST(QLatin1String("DOC:getAmount"), unit2.getAmount(SKGServices::stringToDate(QLatin1String("2007-02-01"))), 7 * 70)
        SKGTEST(QLatin1String("DOC:getAmount"), unit2.getAmount(SKGServices::stringToDate(QLatin1String("2008-01-01"))), 8 * 80)
        SKGTEST(QLatin1String("DOC:getAmount"), unit2.getAmount(SKGServices::stringToDate(QLatin1String("2008-02-01"))), 8 * 80)

        SKGTESTERROR(QLatin1String("UNIT:load"), unit1.load(), true)
        SKGTESTERROR(QLatin1String("UNIT:load"), unit2.load(), true)
        SKGTEST(QLatin1String("DOC:convert"), SKGUnitObject::convert(100, unit1, unit1), 100)
        SKGTEST(QLatin1String("DOC:convert"), SKGUnitObject::convert(100, unit1, unit2), 1.25)
    }

    {
        // Split unit
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        SKGUnitObject unit1(&document1);
        {
            SKGBEGINTRANSACTION(document1, QLatin1String("UNIT"), err)
            SKGTESTERROR(QLatin1String("UNIT:setName"), unit1.setName(QLatin1String("U1")), true)
            SKGTESTERROR(QLatin1String("UNIT:setSymbol"), unit1.setSymbol(QLatin1String("U1")), true)
            SKGTESTERROR(QLatin1String("UNIT:save"), unit1.save(), true)
            SKGTESTERROR(QLatin1String("UNIT:addOrModifyUnitValue"), document1.addOrModifyUnitValue(QLatin1String("U1"), SKGServices::stringToDate(QLatin1String("2001-01-01")), 100), true)
        }
        {
            SKGBEGINTRANSACTION(document1, QLatin1String("UNIT-SPLIT"), err)
            SKGTESTERROR(QLatin1String("DOC:getAmount"), unit1.split(-2), false)
            SKGTESTERROR(QLatin1String("DOC:getAmount"), unit1.split(2), true)
        }

        SKGTESTERROR(QLatin1String("UNIT:load"), unit1.load(), true)
        SKGTEST(QLatin1String("UNIT:getAmount"), unit1.getAmount(SKGServices::stringToDate(QLatin1String("2001-01-01"))), 50)
    }

    {
        // Unit Daily change
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        SKGUnitObject unit1(&document1);
        {
            SKGBEGINTRANSACTION(document1, QLatin1String("UNIT"), err)
            SKGTESTERROR(QLatin1String("UNIT:setName"), unit1.setName(QLatin1String("U1")), true)
            SKGTESTERROR(QLatin1String("UNIT:setSymbol"), unit1.setSymbol(QLatin1String("U1")), true)
            SKGTESTERROR(QLatin1String("UNIT:save"), unit1.save(), true)
            SKGTESTERROR(QLatin1String("UNIT:addOrModifyUnitValue"), document1.addOrModifyUnitValue(QLatin1String("U1"), SKGServices::stringToDate(QLatin1String("2009-01-01")), 100), true)
            SKGTESTERROR(QLatin1String("UNIT:addOrModifyUnitValue"), document1.addOrModifyUnitValue(QLatin1String("U1"), SKGServices::stringToDate(QLatin1String("2009-01-02")), 105), true)
            SKGTESTERROR(QLatin1String("UNIT:addOrModifyUnitValue"), document1.addOrModifyUnitValue(QLatin1String("U1"), SKGServices::stringToDate(QLatin1String("2009-02-01")), 110), true)
        }

        SKGTEST(QLatin1String("UNIT:getDailyChange"), SKGServices::intToString(365 * unit1.getDailyChange(SKGServices::stringToDate(QLatin1String("2009-01-02")))), QLatin1String("1825"))
        SKGTEST(QLatin1String("UNIT:getDailyChange"), SKGServices::intToString(365 * unit1.getDailyChange(SKGServices::stringToDate(QLatin1String("2009-02-01")))), QLatin1String("80"))
    }

    {
        // Unit download with internet code = " " (see https://forum.kde.org/viewtopic.php?f=210&t=163184&p=424630&hilit=skrooge#p424630)
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        SKGUnitObject unit1(&document1);
        {
            SKGBEGINTRANSACTION(document1, QLatin1String("UNIT"), err)
            SKGTESTERROR(QLatin1String("UNIT:setName"), unit1.setName(QLatin1String("DS")), true)
            SKGTESTERROR(QLatin1String("UNIT:setSymbol"), unit1.setSymbol(QLatin1String("DS")), true)
            SKGTESTERROR(QLatin1String("UNIT:setInternetCode"), unit1.setInternetCode(QLatin1String(" ")), true)
            SKGTESTERROR(QLatin1String("UNIT:save"), unit1.save(), true)
            SKGTESTERROR(QLatin1String("UNIT:downloadUnitValue"), unit1.downloadUnitValue(SKGUnitObject::ALL_DAILY, 20), true)
        }
    }

    {
        // Unit download
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        SKGUnitObject unit1(&document1);
        {
            SKGBEGINTRANSACTION(document1, QLatin1String("UNIT"), err)
            SKGTESTERROR(QLatin1String("UNIT:setName"), unit1.setName(QLatin1String("DS")), true)
            SKGTESTERROR(QLatin1String("UNIT:setSymbol"), unit1.setSymbol(QLatin1String("DS")), true)
            SKGTESTERROR(QLatin1String("UNIT:setDownloadSource"), unit1.setDownloadSource(QLatin1String("Boursorama")), true)
            SKGTESTERROR(QLatin1String("UNIT:setInternetCode"), unit1.setInternetCode(QLatin1String("AIRBUS")), true)
            SKGTESTERROR(QLatin1String("UNIT:save"), unit1.save(), true)
            SKGTESTERROR(QLatin1String("UNIT:downloadUnitValue"), unit1.downloadUnitValue(SKGUnitObject::ALL_DAILY, 20), true)

            SKGObjectBase::SKGListSKGObjectBase values;
            SKGTESTERROR(QLatin1String("UNIT:getUnitValues"), unit1.getUnitValues(values), true)
            SKGTESTBOOL("UNIT:getUnitValues", (!values.isEmpty()), true)

            // All download modes
            SKGTESTERROR(QLatin1String("UNIT:downloadUnitValue"), unit1.downloadUnitValue(SKGUnitObject::LAST_WEEKLY, 20), true)
            SKGTESTERROR(QLatin1String("UNIT:downloadUnitValue"), unit1.downloadUnitValue(SKGUnitObject::LAST_DAILY, 20), true)
            SKGTESTERROR(QLatin1String("UNIT:downloadUnitValue"), unit1.downloadUnitValue(SKGUnitObject::LAST_MONTHLY, 20), true)
            SKGTESTERROR(QLatin1String("UNIT:downloadUnitValue"), unit1.downloadUnitValue(SKGUnitObject::ALL_MONTHLY, 20), true)
            SKGTESTERROR(QLatin1String("UNIT:downloadUnitValue"), unit1.downloadUnitValue(SKGUnitObject::ALL_WEEKLY, 20), true)

            SKGTESTERROR(QLatin1String("UNIT:getUnitValues"), unit1.getUnitValues(values), true)
            SKGTESTBOOL("UNIT:getUnitValues", static_cast<unsigned int>(values.count() > 1), static_cast<unsigned int>(true))

            QUrl url;
            unit1.getUrl(url);  // For coverage
        }
    }
    {
        // Unit download
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        SKGUnitObject unit1(&document1);
        {
            SKGBEGINTRANSACTION(document1, QLatin1String("UNIT"), err)
            SKGTESTERROR(QLatin1String("UNIT:setName"), unit1.setName(QLatin1String("FORMULA")), true)
            SKGTESTERROR(QLatin1String("UNIT:setSymbol"), unit1.setSymbol(QLatin1String("FORMULA")), true)
            SKGTESTERROR(QLatin1String("UNIT:setInternetCode"), unit1.setInternetCode(QLatin1String("=5")), true)
            SKGTESTERROR(QLatin1String("UNIT:save"), unit1.save(), true)
            SKGTESTERROR(QLatin1String("UNIT:downloadUnitValue"), unit1.downloadUnitValue(SKGUnitObject::ALL_DAILY, 20), true)

            SKGObjectBase::SKGListSKGObjectBase values;
            SKGTESTERROR(QLatin1String("UNIT:getUnitValues"), unit1.getUnitValues(values), true)
            SKGTESTBOOL("UNIT:getUnitValues.isEmpty", values.isEmpty(), false)
        }
    }
    {
        // Unit download
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;

        {
            SKGBEGINTRANSACTION(document1, QLatin1String("UNIT"), err)
            SKGUnitObject euro;
            SKGTESTERROR(QLatin1String("UNIT:createCurrencyUnit"), SKGUnitObject::createCurrencyUnit(&document1, QLatin1String("EUR"), euro), true)
            SKGUnitObject fr;
            SKGTESTERROR(QLatin1String("UNIT:createCurrencyUnit"), SKGUnitObject::createCurrencyUnit(&document1, QLatin1String("GBP"), fr), true)
            SKGTESTERROR(QLatin1String("UNIT:downloadUnitValue"), fr.downloadUnitValue(SKGUnitObject::LAST), true)

            SKGObjectBase::SKGListSKGObjectBase values;
            SKGTESTERROR(QLatin1String("UNIT:getUnitValues"), fr.getUnitValues(values), true)
            SKGTESTBOOL("UNIT:getUnitValues", values.isEmpty(), false)
        }
    }
    {
        // Unit simplify
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        SKGUnitObject unit1(&document1);
        {
            SKGBEGINTRANSACTION(document1, QLatin1String("UNIT"), err)
            SKGTESTERROR(QLatin1String("UNIT:setName"), unit1.setName(QLatin1String("U1")), true)
            SKGTESTERROR(QLatin1String("UNIT:setSymbol"), unit1.setSymbol(QLatin1String("U1")), true)
            SKGTESTERROR(QLatin1String("UNIT:save"), unit1.save(), true)

            SKGTESTERROR(QLatin1String("UNIT:addOrModifyUnitValue"), document1.addOrModifyUnitValue(QLatin1String("U1"), QDate::currentDate().addDays(-7), 1), true)
            SKGTESTERROR(QLatin1String("UNIT:addOrModifyUnitValue"), document1.addOrModifyUnitValue(QLatin1String("U1"), QDate::currentDate().addDays(-6), 2), true)
            SKGTESTERROR(QLatin1String("UNIT:addOrModifyUnitValue"), document1.addOrModifyUnitValue(QLatin1String("U1"), QDate::currentDate().addDays(-5), 3), true)

            SKGTESTERROR(QLatin1String("UNIT:addOrModifyUnitValue"), document1.addOrModifyUnitValue(QLatin1String("U1"), QDate::currentDate().addMonths(-3).addDays(-7), 1), true)
            SKGTESTERROR(QLatin1String("UNIT:addOrModifyUnitValue"), document1.addOrModifyUnitValue(QLatin1String("U1"), QDate::currentDate().addMonths(-3).addDays(-6), 2), true)
            SKGTESTERROR(QLatin1String("UNIT:addOrModifyUnitValue"), document1.addOrModifyUnitValue(QLatin1String("U1"), QDate::currentDate().addMonths(-3).addDays(-5), 3), true)

            SKGTESTERROR(QLatin1String("UNIT:addOrModifyUnitValue"), document1.addOrModifyUnitValue(QLatin1String("U1"), QDate::currentDate().addYears(-1).addDays(-7), 1), true)
            SKGTESTERROR(QLatin1String("UNIT:addOrModifyUnitValue"), document1.addOrModifyUnitValue(QLatin1String("U1"), QDate::currentDate().addMonths(-1).addDays(-6), 2), true)
            SKGTESTERROR(QLatin1String("UNIT:addOrModifyUnitValue"), document1.addOrModifyUnitValue(QLatin1String("U1"), QDate::currentDate().addMonths(-1).addDays(-5), 3), true)

            SKGTESTERROR(QLatin1String("UNIT:addOrModifyUnitValue"), document1.addOrModifyUnitValue(QLatin1String("U1"), SKGServices::stringToDate(QLatin1String("2009-01-01")), 1), true)
            SKGTESTERROR(QLatin1String("UNIT:addOrModifyUnitValue"), document1.addOrModifyUnitValue(QLatin1String("U1"), SKGServices::stringToDate(QLatin1String("2009-01-02")), 2), true)
            SKGTESTERROR(QLatin1String("UNIT:addOrModifyUnitValue"), document1.addOrModifyUnitValue(QLatin1String("U1"), SKGServices::stringToDate(QLatin1String("2009-01-03")), 3), true)
        }

        {
            SKGBEGINTRANSACTION(document1, QLatin1String("SYMPLIFY"), err)
            SKGTESTERROR(QLatin1String("UNIT:setName"), unit1.simplify(), true)
        }
        SKGObjectBase::SKGListSKGObjectBase oUnitValueList;
        SKGTESTERROR(QLatin1String("UNIT:getUnitValues"), unit1.getUnitValues(oUnitValueList), true)
        SKGTESTBOOL("UNIT:oUnitValueList", (oUnitValueList.count() == 10 || oUnitValueList.count() == 11), true)
    }

    {
        SKGTESTERROR(QLatin1String("UNIT:addSource"), SKGUnitObject::addSource(QLatin1String("test"), false), true)
        SKGTESTBOOL("UNIT:isWritable", SKGUnitObject::isWritable(QLatin1String("test")), true)
        SKGTESTERROR(QLatin1String("UNIT:deleteSource"), SKGUnitObject::deleteSource(QLatin1String("test")), true)
        SKGTESTERROR(QLatin1String("UNIT:deleteSource"), SKGUnitObject::deleteSource(QLatin1String("notfound")), false)
    }
    // End test
    SKGENDTEST()
}
