/* Copyright 1997-2006 Red Hat, Inc.
 *
 * This software may be freely redistributed under the terms of the GNU
 * public license.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Thu Jan 27 01:00:12 PST 2000 - <thumper@alumni.caltech.edu>
 * - made changes to support "configure" script
 * - made changes to allow --disable-nls from configure
 */
#include "config.h"
#include <alloca.h>
#include <ctype.h>
#include <dirent.h>
#include <errno.h>
#include <newt.h>

#if (!HAVE_LIBPOPT && !HAVE_GETOPT_LONG && !HAVE_LIBGETOPT)
    #error "Either the Redhat popt or GNU getopt library must be installed."
#endif
#if HAVE_LIBPOPT
    #include <popt.h>
#else /* !HAVE_LIBPOPT */
    /* we need GNU getopt-long */
    #include <getopt.h>
#endif /* HAVE_LIBPOPT */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>
#include <locale.h> 
#include <libintl.h> 

#define _(String) gettext((String)) 

#include "leveldb.h"


struct config conf;		// used by leveldb.c


/* return 1 on cancel, 2 on error, 0 on success */
static int servicesWindow(struct service * services, int numServices,
			  int levels, int backButton) {
    newtComponent label, subform, ok, cancel;
    newtComponent * checkboxes, form, curr, blank;
    newtComponent sb = NULL;
    newtGrid grid, subgrid, buttons;
    char * states;
    int i, done = 0, update = 0, j;
    int width, height;
    struct newtExitStruct e;

    newtPushHelpLine(_("Press <F1> for more information on a service."));

    newtGetScreenSize(&width, &height);
    width = width > 80 ? width - 60 : 20;
    height = height > 25 ? height - 17 : 8;
	
	
    sb = newtVerticalScrollbar(-1, -1, height, NEWT_COLORSET_CHECKBOX,
				    NEWT_COLORSET_ACTCHECKBOX);

    subform = newtForm(sb, NULL, 0);
    newtFormSetBackground(subform, NEWT_COLORSET_CHECKBOX);
    newtFormSetHeight(subform, height);

    checkboxes = alloca(sizeof(*checkboxes) * numServices);
    states = alloca(sizeof(*states) * numServices);
    
    for (i = 0; i < numServices; i++) {
	if (services[i].type == TYPE_XINETD) {
		checkboxes[i] = newtCheckbox(-1, i, services[i].name, 
				     services[i].levels ? '*' : ' ', NULL, 
				     states + i);
	} else {
		for (j = 0; j <= conf.maxlevel; j++) {
			if (levels & (1 << j)) {
				if (isOn(services[i].name, j)) break;
			}
		}
		checkboxes[i] = newtCheckbox(-1, i, services[i].name, 
					     (j != (conf.maxlevel+1)) ?
							'*' : ' ', NULL, 
					     states + i);
	}
	newtFormAddComponent(subform, checkboxes[i]);
    }

    newtFormSetWidth(subform, width);

    buttons = newtButtonBar(_("Ok"), &ok, backButton ? _("Back") : _("Cancel"),
			    &cancel, NULL);

    blank = newtForm(NULL, NULL, 0);
    newtFormSetWidth(blank, 2);
    newtFormSetHeight(blank, height);
    newtFormSetBackground(blank, NEWT_COLORSET_CHECKBOX);

    subgrid = newtGridHCloseStacked(NEWT_GRID_COMPONENT, subform,
			            NEWT_GRID_COMPONENT, blank,
			            NEWT_GRID_COMPONENT, sb, NULL);

    label = newtTextboxReflowed(-1, -1,  _("What services should be "
				"automatically started?"), 30, 0, 20, 0);
    grid = newtGridBasicWindow(label, subgrid, buttons);

    form = newtForm(NULL, NULL, 0);
    newtGridAddComponentsToForm(grid, form, 1);
    newtGridWrappedWindow(grid, _("Services"));
    newtGridFree(grid, 1);

    newtFormAddHotKey(form, NEWT_KEY_F1);

    while (!done) {
	newtFormRun(form, &e);

	if (e.reason == NEWT_EXIT_HOTKEY) {
	    if (e.u.key == NEWT_KEY_F12) {
		done = 1;
		update = 1;
	    } else {
		/* must be F1 */
		curr = newtFormGetCurrent(subform);
		for (i = 0; i < numServices; i++)
		    if (curr == checkboxes[i]) break;

		if (i < numServices && services[i].desc) 
		    newtWinMessage(services[i].name, _("Ok"), services[i].desc);
	    }
	} else {
	    done = 1;
	    update = (e.u.co == ok);
	}
    }
    
    newtPopWindow();
    newtFormDestroy(form);
    
    if (!update) return 1;

    for (i = 0; i < numServices; i++) {
      if (services[i].type == TYPE_XINETD) {
        if ((services[i].enabled && states[i] != '*') ||
	    (!services[i].enabled && states[i] == '*'))
	      setXinetdService(services[i], states[i] == '*');
      } else {
	      for (j = 0; j <= conf.maxlevel; j++) {
		      if (levels & (1 << j))
			doSetService(services[i], j, states[i] == '*');
	      }
      }
    }

    return 0;
}

static int serviceNameCmp(const void * a, const void * b) {
    const struct service * first = a;
    const struct service * second = b;

    return strcmp(first->name, second->name);
}

static int getServices(struct service ** servicesPtr, int * numServicesPtr,
		       int backButton, int honorHide) {
    DIR * dir;
    struct dirent * ent;
    struct stat sb;
    char fn[1024];
    struct service * services;
    int numServices = 0;
    int numServicesAlloced, rc;
    int err = 0;
    char xinetd[250];

    numServicesAlloced = 10;
    services = malloc(sizeof(*services) * numServicesAlloced);

    if (!(dir = opendir(conf.initdir))) {
	fprintf(stderr, "failed to open %s: %s\n", conf.initdir,
		strerror(errno));
        return 2;
    }

    while ((ent = readdir(dir))) {
	if (strchr(ent->d_name, '~') || strchr(ent->d_name, ',') ||
	    strchr(ent->d_name, '.')) continue;

	sprintf(fn, "%s/%s", conf.initdir, ent->d_name);
	if (stat(fn, &sb))
	{
		fprintf(stderr, _("error reading info for service %s: %s\n"),
			ent->d_name, strerror(errno));
		continue;
	}
	if (!S_ISREG(sb.st_mode)) continue;

	if (numServices == numServicesAlloced) {
	    numServicesAlloced += 10;
	    services = realloc(services, 
				numServicesAlloced * sizeof(*services));
	}

	rc = readServiceInfo(ent->d_name, services + numServices, honorHide);
        
	if (!rc) {
		int i;
		
		rc = -2;
		for (i = 0 ; i < 7 ; i++) {
			if (isConfigured(ent->d_name, i)) {
				rc = 0;
				break;
			}
		}
	}
	 
	if (rc == -1) {
	    fprintf(stderr, _("error reading info for service %s: %s\n"),
			ent->d_name, strerror(errno));
	    closedir(dir);
	    return 2;
	} else if (!rc)
	    numServices++;
    }

    closedir(dir);

    snprintf(xinetd, sizeof(xinetd),
	    "%s/xinetd", conf.initdir);
    if (!stat(xinetd,&sb)) {
	if (!(dir = opendir(conf.xinetddir))) {
	    fprintf(stderr, "failed to open %s: %s\n", conf.xinetddir,
		    strerror(errno));
	    return 2;
	}

	while ((ent = readdir(dir))) {
	    if (strchr(ent->d_name, '~') || strchr(ent->d_name, ',') ||
		strchr(ent->d_name, '.')) continue;

	    sprintf(fn, "%s/%s", conf.xinetddir, ent->d_name);
	    if (stat(fn, &sb))
	    {
		    err = errno;
		    continue;
	    }
	    if (!S_ISREG(sb.st_mode)) continue;

	    if (numServices == numServicesAlloced) {
		numServicesAlloced += 10;
		services = realloc(services, 
				    numServicesAlloced * sizeof(*services));
	    }

	    rc = readXinetdServiceInfo(ent->d_name, services + numServices, honorHide);
	    
	    if (rc == -1) {
		fprintf(stderr, _("error reading info for service %s: %s\n"),
			    ent->d_name, strerror(errno));
		closedir(dir);
		return 2;
	    } else if (!rc)
		numServices++;
	}

	if (err) {
	    fprintf(stderr, _("error reading from directory %s: %s\n"),
		    conf.xinetddir, strerror(err));
	    return 1;
	}
    }

    qsort(services, numServices, sizeof(*services), serviceNameCmp);

    *servicesPtr = services;
    *numServicesPtr = numServices;

    return 0;
}

int main(int argc, char ** argv) {
    struct service * services;
    int numServices;
    int levels = -1;
    char * levelsStr = NULL;
    char *progName;
    int rc, backButton = 0, hide = 0;
#if HAVE_LIBPOPT
    poptContext optCon;
    struct poptOption optionsTable[] = {
	    { "back", '\0', 0, &backButton, 0 },
	    { "level", '\0', POPT_ARG_STRING, &levelsStr, 0 },
	    { "rc.d", '\0', POPT_ARG_STRING, &conf.rcdir, 0 },
	    { "init.d", '\0', POPT_ARG_STRING, &conf.initdir, 0 },
	    { "xinetd.d", '\0', POPT_ARG_STRING, &conf.xinetddir, 0 },
	    { "hide", '\0', 0, &hide, 0 },
	    { 0, 0, 0, 0, 0 } 
    };
#else /* !HAVE_LIBPOPT */
    int i;
    struct option optionsTable[] = 
    {
        { "back", 0, &backButton, 1 },
        { "level", 1, 0, 0 },
        { "rc.d", 1, 0, 0 },
        { "init.d", 1, 0, 0 },
        { "xinetd.d", 1, 0, 0 },
        { "hide", 0, &hide, 1 },
        { 0, 0, 0, 0 }
    };
#endif /* HAVE_LIBPOPT */

    conf.rcdir = RCDIR;		// default to value found by configure
    conf.initdir = INITDIR;
    conf.xinetddir = XINETDDIR;
    conf.maxlevel = MAXLEVEL;

    setlocale(LC_ALL, ""); 
    bindtextdomain(PACKAGE, LOCALEDIR); 
    textdomain(PACKAGE); 

    progName = argv[0];

    if (getuid() != 0) {
	fprintf(stderr, _("You must be root to run %s.\n"),progName);
	exit(1);
    }
    
#if HAVE_LIBPOPT
    optCon = poptGetContext(progName, argc, argv, optionsTable, 0);
    poptReadDefaultConfig(optCon, 1);

    if ((rc = poptGetNextOpt(optCon)) < -1) {
	fprintf(stderr, "%s: %s\n", 
		poptBadOption(optCon, POPT_BADOPTION_NOALIAS), 
		poptStrerror(rc));
	exit(1);
    }
#else /* !HAVE_LIBPOPT */
    /* turn off getopt_long from reporting errors on stderr */
    opterr = 0;

    /* process all the options so that we know what the user wants */
    while ((rc = getopt_long(argc, argv, "",
				    optionsTable, &i)) != -1) {
	switch (rc) {
	    case ':':	/* missing arg for option  -- fall through */
	    case '?':	/* extra arg for option */
			exit(-1);
	    case 0:	/* found a long option... check for an arg */
			switch (i) {
			    /* these numbers correspond to the
			     * index in optionsTable[]. */
			    case 1: levelsStr = optarg; break;
			    case 2: conf.rcdir = optarg; break;
			    case 3: conf.initdir = optarg; break;
			    case 4: conf.xinetddir = optarg; break;
			    default: break;	/* no arg */
			}
			break;
	    default:
		fprintf(stderr, "%s: weird, getopt_long = %d (%c), i = %d\n",
			progName, rc, (char) rc, i);
		fprintf(stderr, "Terminating.\n");
		exit(1);
	}
    }
#endif /* HAVE_LIBPOPT */

    if (levelsStr) {
	levels = parseLevels(levelsStr, 0);
	if (levels == -1) {
	    fprintf(stderr, _("bad argument to --levels\n"));
	    exit(2);
	}
    }

    if (getServices(&services, &numServices, backButton, hide)) return 1;
    if (!numServices) {
	fprintf(stderr, _("No services may be managed by ntsysv!\n"));
	return 2;
    }

    newtInit();
    newtCls();

    newtPushHelpLine(NULL);
    newtDrawRootText(0, 0,
		     "ntsysv " VERSION " - (C) 2000-2001 Red Hat, Inc. ");
   
    if (levels==-1) 
     levels=(1<<currentRunlevel());

    rc = servicesWindow(services, numServices, levels, backButton);

    newtFinished();

    return rc;
}
