// This may look like C code, but it is really -*- C++ -*-

/* 
Copyright (C) 1988 Free Software Foundation
    written by Doug Lea (dl@rocky.oswego.edu)

This file is part of GNU CC.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the GNU CC General Public
License for full details.

Everyone is granted permission to copy, modify and redistribute
GNU CC, but only under the conditions described in the
GNU CC General Public License.   A copy of this license is
supposed to have been given to you along with GNU CC so you
can know your rights and responsibilities.  It should be in a
file named COPYING.  Among other things, the copyright notice
and this notice must be preserved on all copies.  
*/

#ifndef _Rational_h
#define _Rational_h 1

#include <Integer.h>
#include <math.h>

class Rational
{
  Integer          num;
  Integer          den;


public:
                   Rational();
                   Rational(double);
                   Rational(long n, long d = 1);
                   Rational(Integer& n);
                   Rational(Integer& n, Integer& d);
                   Rational(Rational&);

                   ~Rational();

  Rational&        operator =  (Rational& y);

  friend int       operator == (Rational& x, Rational& y);
  friend int       operator != (Rational& x, Rational& y);
  friend int       operator <  (Rational& x, Rational& y);
  friend int       operator <= (Rational& x, Rational& y);
  friend int       operator >  (Rational& x, Rational& y);
  friend int       operator >= (Rational& x, Rational& y);

  friend Rational  operator +  (Rational& x, Rational& y);
  friend Rational  operator -  (Rational& x, Rational& y);
  friend Rational  operator *  (Rational& x, Rational& y);
  friend Rational  operator /  (Rational& x, Rational& y);
  friend Rational& operator <? (Rational& x, Rational& y); // min
  friend Rational& operator >? (Rational& x, Rational& y); // max

  Rational         operator -  ();

  Rational&        operator += (Rational& y);
  Rational&        operator -= (Rational& y);
  Rational&        operator *= (Rational& y);
  Rational&        operator /= (Rational& y);

// builtin Rational functions


  void             negate();                      // x = -x
  void             invert();                      // x = 1/x

  friend int       sign(Rational& x);             // -1, 0, or +1
  friend Rational  abs(Rational& x);              // absolute value
  friend Rational  sqr(Rational& x);              // square
  friend Rational  pow(Rational& x, long y);
  friend Rational  pow(Rational& x, Integer& y);
  Integer          numerator();
  Integer          denominator();

// coercion & conversion

  double           operator double();
  friend Integer   floor(Rational& x);
  friend Integer   ceil(Rational& x);
  friend Integer   trunc(Rational& x);
  friend Integer   round(Rational& x);

  friend istream&  operator >> (istream& s, Rational& y);
  friend ostream&  operator << (ostream& s, Rational& y);

// miscellany

  friend int       compare(Rational& x, Rational& y);
  void             normalize();
  void             error(char* msg);
};


// error handlers

extern  void default_Rational_error_handler(char*);
extern  one_arg_error_handler_t Rational_error_handler;

extern  one_arg_error_handler_t 
        set_Rational_error_handler(one_arg_error_handler_t f);


//#ifdef __OPTIMIZE__

inline Rational::Rational()  {}
inline Rational::~Rational() {}

inline Rational::Rational(Rational& y)
{
  num = y.num; den = y.den;
}

inline Rational::Rational(Integer& n)
{
  num = n; den = 1;
}

inline Rational::Rational(Integer& n, Integer& d)
{
  num = n; den = d; normalize();
}

inline Rational::Rational(long n, long d = 1)
{
  num = n; den = d; normalize();
}

inline  Rational& Rational::operator =  (Rational& y)
{
  num = y.num;  den = y.den;   return *this;
}

inline Integer Rational::numerator()
{
  return num;
}

inline Integer Rational::denominator()
{
  return den;
}

inline int operator == (Rational& x, Rational& y)
{
  return compare(x.num, y.num) == 0 && compare(x.den, y.den) == 0;
}

inline int operator != (Rational& x, Rational& y)
{
  return compare(x.num, y.num) != 0 || compare(x.den, y.den) != 0;
}

inline int operator <  (Rational& x, Rational& y)
{
  return compare(x, y) <  0; 
}

inline int operator <= (Rational& x, Rational& y)
{
  return compare(x, y) <= 0; 
}

inline int operator >  (Rational& x, Rational& y)
{
  return compare(x, y) >  0; 
}


inline int operator >= (Rational& x, Rational& y)
{
  return compare(x, y) >= 0; 
}

inline int sign(Rational& x)
{
  return sign(x.num);
}

inline void Rational::negate()
{
  num.negate();
}


inline Rational& operator <? (Rational& x, Rational& y)
{
  if (compare(x, y) <= 0) return x; else return y;
}

inline Rational& operator >? (Rational& x, Rational& y)
{
  if (compare(x, y) >= 0) return x; else return y;
}

inline Integer trunc(Rational& x)
{
  return x.num / x.den ;
}

inline Rational pow(Rational& x, Integer& y)
{
  return pow(x, long(y));       // correct since max bits < max long
}                               // & num & den are relative primes

//#endif

#endif
