/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.sis.referencing.operation.provider;

import jakarta.xml.bind.annotation.XmlTransient;
import org.opengis.util.FactoryException;
import org.opengis.parameter.ParameterDescriptor;
import org.opengis.parameter.ParameterDescriptorGroup;
import org.opengis.referencing.cs.CoordinateSystem;
import org.opengis.referencing.operation.Conversion;
import org.opengis.referencing.operation.MathTransform;
import org.apache.sis.metadata.iso.citation.Citations;
import org.apache.sis.parameter.ParameterBuilder;
import org.apache.sis.parameter.Parameters;
import org.apache.sis.util.internal.shared.Constants;
import org.apache.sis.measure.Units;


/**
 * The provider for <q>Geographic 2D to 3D conversion</q>.
 * The default operation sets the ellipsoidal height to zero.
 *
 * <p>This operation is a SIS extension; the EPSG dataset defines only the 3D to 2D case.
 * Consequently, WKT formatting will not represent "2D to 3D" transform. Instead, WKT will
 * format the inverse ({@code "INVERSE_MT"}) of 3D to 2D transform.</p>
 *
 * @author  Martin Desruisseaux (Geomatys)
 *
 * @see Geographic3Dto2D
 * @see Spherical2Dto3D
 */
@XmlTransient
public final class Geographic2Dto3D extends AbstractProvider {
    /**
     * Serial number for inter-operability with different versions.
     */
    private static final long serialVersionUID = -1198461394243672064L;

    /**
     * The name used by Apache <abbr>SIS</abbr> for this operation method.
     */
    public static final String NAME = "Geographic2D to 3D conversion";

    /**
     * The ellipsoidal height to set.
     *
     * <!-- Generated by ParameterNameTableGenerator -->
     * <table class="sis">
     *   <caption>Parameter names</caption>
     *   <tr><td> SIS:     </td><td> height </td></tr>
     * </table>
     */
    private static final ParameterDescriptor<Double> HEIGHT;

    /**
     * The default height value.
     */
    public static final double DEFAULT_HEIGHT = 0;

    /**
     * The group of all parameters expected by this coordinate operation.
     */
    public static final ParameterDescriptorGroup PARAMETERS;
    static {
        final ParameterBuilder builder = builder().setCodeSpace(Citations.SIS, Constants.SIS);
        HEIGHT = builder.addName("height").create(DEFAULT_HEIGHT, Units.METRE);
        PARAMETERS = builder.addName(NAME).createGroup(HEIGHT);
    }

    /**
     * The canonical instance of this operation method.
     *
     * @see #provider()
     */
    private static final Geographic2Dto3D INSTANCE = new Geographic2Dto3D();

    /**
     * Returns the canonical instance of this operation method.
     * This method is invoked by {@link java.util.ServiceLoader} using reflection.
     *
     * @return the canonical instance of this operation method.
     */
    public static Geographic2Dto3D provider() {
        return INSTANCE;
    }

    /**
     * Creates a new provider.
     *
     * @todo Make this constructor private after we stop class-path support.
     *       Instantiate {@code Geographic3Dto2D} directly with the parameters.
     *       Modify this {@code Geographic2Dto3D} so that it does not extent anything.
     */
    public Geographic2Dto3D() {
        super(Conversion.class, PARAMETERS,
              CoordinateSystem.class, false,
              CoordinateSystem.class, false,
              (byte) 2);
    }

    /**
     * Returns the inverse of this operation.
     */
    @Override
    public AbstractProvider inverse() {
        return Geographic3Dto2D.provider();
    }

    /**
     * Returns the operation method which is the closest match for the given transform.
     * This is an adjustment based on the number of dimensions only, on the assumption
     * that the given transform has been created by this provider or a compatible one.
     */
    @Override
    public AbstractProvider variantFor(final MathTransform transform) {
        return transform.getSourceDimensions() > transform.getTargetDimensions() ? Geographic3Dto2D.provider() : this;
    }

    /**
     * Creates the transform adding a constant ellipsoidal height.
     * The parameter value is unconditionally converted to metres.
     *
     * @param  context  the parameter values together with its context.
     * @return the math transform for the given parameter values.
     * @throws FactoryException if an error occurred while creating the transform.
     */
    @Override
    public MathTransform createMathTransform(final Context context) throws FactoryException {
        final Parameters pg = Parameters.castOrWrap(context.getCompletedParameters());
        return Geographic3Dto2D.createMathTransform(context,
                context.getSourceDimensions().orElse(2),
                context.getTargetDimensions().orElse(3),
                pg.doubleValue(HEIGHT));
    }
}
