// Copyright 2019 Drone IO, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package users

import (
	"encoding/json"
	"net/http"
	"time"

	"github.com/dchest/uniuri"
	"github.com/drone/drone/core"
	"github.com/drone/drone/handler/api/render"
	"github.com/drone/drone/handler/api/request"
	"github.com/drone/drone/logger"
)

type userWithToken struct {
	*core.User
	Token string `json:"token"`
}

// HandleCreate returns an http.HandlerFunc that processes an http.Request
// to create the named user account in the system.
func HandleCreate(users core.UserStore, service core.UserService, sender core.WebhookSender) http.HandlerFunc {
	return func(w http.ResponseWriter, r *http.Request) {
		in := new(userWithToken)
		err := json.NewDecoder(r.Body).Decode(in)
		if err != nil {
			render.BadRequest(w, err)
			logger.FromRequest(r).WithError(err).
				Debugln("api: cannot unmarshal request body")
			return
		}

		user := &core.User{
			Login:   in.Login,
			Active:  true,
			Admin:   in.Admin,
			Machine: in.Machine,
			Created: time.Now().Unix(),
			Updated: time.Now().Unix(),
			Hash:    in.Token,
		}
		if user.Hash == "" {
			user.Hash = uniuri.NewLen(32)
		}

		// if the user is not a machine account, we lookup
		// the user in the remote system. We can then augment
		// the user input with the remote system data.
		if !user.Machine {
			viewer, _ := request.UserFrom(r.Context())
			remote, err := service.FindLogin(r.Context(), viewer, user.Login)
			if err == nil {
				if user.Login != remote.Login && remote.Login != "" {
					user.Login = remote.Login
				}
				if user.Email == "" {
					user.Email = remote.Email
				}
			}
		}

		err = user.Validate()
		if err != nil {
			render.ErrorCode(w, err, 400)
			logger.FromRequest(r).WithError(err).
				Errorln("api: invalid username")
			return
		}

		err = users.Create(r.Context(), user)
		if err == core.ErrUserLimit {
			render.ErrorCode(w, err, 402)
			logger.FromRequest(r).WithError(err).
				Errorln("api: cannot create user")
			return
		}
		if err != nil {
			render.InternalError(w, err)
			logger.FromRequest(r).WithError(err).
				Warnln("api: cannot create user")
			return
		}

		err = sender.Send(r.Context(), &core.WebhookData{
			Event:  core.WebhookEventUser,
			Action: core.WebhookActionCreated,
			User:   user,
		})
		if err != nil {
			logger.FromRequest(r).WithError(err).
				Warnln("api: cannot send webhook")
		}

		var out interface{} = user
		// if the user is a machine account the api token
		// is included in the response.
		if user.Machine {
			out = &userWithToken{user, user.Hash}
		}
		render.JSON(w, out, 200)
	}
}
