/*
 * cafile.c
 * vim: expandtab:ts=4:sts=4:sw=4
 *
 * Copyright (C) 2022 Steffen Jaeckel <jaeckel-floss@eyet-services.de>
 *
 * This file is part of Profanity.
 *
 * Profanity is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Profanity is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Profanity.  If not, see <https://www.gnu.org/licenses/>.
 *
 * In addition, as a special exception, the copyright holders give permission to
 * link the code of portions of this program with the OpenSSL library under
 * certain conditions as described in each individual source file, and
 * distribute linked combinations including the two.
 *
 * You must obey the GNU General Public License in all respects for all of the
 * code used other than OpenSSL. If you modify file(s) with this exception, you
 * may extend this exception to your version of the file(s), but you are not
 * obligated to do so. If you do not wish to do so, delete this exception
 * statement from your version. If you delete this exception statement from all
 * source files in the program, then also delete it here.
 *
 */

#include "config.h"

#include <fcntl.h>
#include <glib.h>
#include <errno.h>
#include <string.h>
#include <sys/wait.h>

#include "common.h"
#include "config/files.h"
#include "log.h"
#include "xmpp/xmpp.h"

static gchar*
_cafile_name(void)
{
    gchar* certs_dir = files_get_data_path(DIR_CERTS);
    if (!create_dir(certs_dir)) {
        g_free(certs_dir);
        return NULL;
    }
    gchar* filename = g_strdup_printf("%s/CAfile.pem", certs_dir);
    g_free(certs_dir);
    return filename;
}

void
cafile_add(const TLSCertificate* cert)
{
    if (!cert->pem) {
        log_error("[CAfile] can't store cert with fingerprint %s: PEM is empty", cert->fingerprint);
        return;
    }
    gchar* cafile = _cafile_name();
    if (!cafile)
        return;
    gchar *contents = NULL, *new_contents = NULL;
    gsize length;
    GError* glib_error = NULL;
    if (g_file_test(cafile, G_FILE_TEST_EXISTS)) {
        if (!g_file_get_contents(cafile, &contents, &length, &glib_error)) {
            log_error("[CAfile] could not read from %s: %s", cafile, glib_error ? glib_error->message : "No GLib error given");
            goto out;
        }
        if (strstr(contents, cert->fingerprint)) {
            log_debug("[CAfile] fingerprint %s already stored", cert->fingerprint);
            goto out;
        }
    }
    const char* header = "# Profanity CAfile\n# DO NOT EDIT - this file is automatically generated";
    new_contents = g_strdup_printf("%s\n\n# %s\n%s", contents ? contents : header, cert->fingerprint, cert->pem);
    if (!g_file_set_contents(cafile, new_contents, -1, &glib_error))
        log_error("[CAfile] could not write to %s: %s", cafile, glib_error ? glib_error->message : "No GLib error given");
out:
    g_free(new_contents);
    g_free(contents);
    g_free(cafile);
}

gchar*
cafile_get_name(void)
{
    gchar* cafile = _cafile_name();
    if (!g_file_test(cafile, G_FILE_TEST_EXISTS)) {
        /* That's no problem!
         * There's no need to have a profanity-specific CAfile if all CA's
         * of servers you're trying to connect to are in your OS trust-store
         */
        log_debug("[CAfile] file %s not created yet", cafile);
        g_free(cafile);
        cafile = NULL;
    }
    return cafile;
}
