/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
/** IDs are delimited by an empty space, as per the spec. */
const ID_DELIMITER = ' ';
/**
 * Adds the given ID to the specified ARIA attribute on an element.
 * Used for attributes such as aria-labelledby, aria-owns, etc.
 */
export function addAriaReferencedId(el, attr, id) {
    const ids = getAriaReferenceIds(el, attr);
    if (ids.some(existingId => existingId.trim() == id.trim())) {
        return;
    }
    ids.push(id.trim());
    el.setAttribute(attr, ids.join(ID_DELIMITER));
}
/**
 * Removes the given ID from the specified ARIA attribute on an element.
 * Used for attributes such as aria-labelledby, aria-owns, etc.
 */
export function removeAriaReferencedId(el, attr, id) {
    const ids = getAriaReferenceIds(el, attr);
    const filteredIds = ids.filter(val => val != id.trim());
    if (filteredIds.length) {
        el.setAttribute(attr, filteredIds.join(ID_DELIMITER));
    }
    else {
        el.removeAttribute(attr);
    }
}
/**
 * Gets the list of IDs referenced by the given ARIA attribute on an element.
 * Used for attributes such as aria-labelledby, aria-owns, etc.
 */
export function getAriaReferenceIds(el, attr) {
    // Get string array of all individual ids (whitespace delimited) in the attribute value
    return (el.getAttribute(attr) || '').match(/\S+/g) || [];
}
//# sourceMappingURL=data:application/json;base64,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