"use strict";
/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getStartOffsetOfAttribute = exports.findAttributeOnElementWithAttrs = exports.findAttributeOnElementWithTag = exports.findElementsWithAttribute = void 0;
const parse5_1 = require("parse5");
/**
 * Parses a HTML fragment and traverses all AST nodes in order find elements that
 * include the specified attribute.
 */
function findElementsWithAttribute(html, attributeName) {
    const document = (0, parse5_1.parseFragment)(html, { sourceCodeLocationInfo: true });
    const elements = [];
    const visitNodes = (nodes) => {
        nodes.forEach(n => {
            const node = n;
            if (node.childNodes) {
                visitNodes(node.childNodes);
            }
            if (node.attrs?.some(attr => attr.name === attributeName.toLowerCase())) {
                elements.push(node);
            }
        });
    };
    visitNodes(document.childNodes);
    return elements;
}
exports.findElementsWithAttribute = findElementsWithAttribute;
/**
 * Finds elements with explicit tag names that also contain the specified attribute. Returns the
 * attribute start offset based on the specified HTML.
 */
function findAttributeOnElementWithTag(html, name, tagNames) {
    return findElementsWithAttribute(html, name)
        .filter(element => tagNames.includes(element.tagName))
        .map(element => getStartOffsetOfAttribute(element, name));
}
exports.findAttributeOnElementWithTag = findAttributeOnElementWithTag;
/**
 * Finds elements that contain the given attribute and contain at least one of the other
 * specified attributes. Returns the primary attribute's start offset based on the specified HTML.
 */
function findAttributeOnElementWithAttrs(html, name, attrs) {
    return findElementsWithAttribute(html, name)
        .filter(element => attrs.some(attr => hasElementAttribute(element, attr)))
        .map(element => getStartOffsetOfAttribute(element, name));
}
exports.findAttributeOnElementWithAttrs = findAttributeOnElementWithAttrs;
/** Shorthand function that checks if the specified element contains the given attribute. */
function hasElementAttribute(element, attributeName) {
    return element.attrs && element.attrs.some(attr => attr.name === attributeName.toLowerCase());
}
/** Gets the start offset of the given attribute from a Parse5 element. */
function getStartOffsetOfAttribute(element, attributeName) {
    return element.sourceCodeLocation.attrs[attributeName.toLowerCase()].startOffset;
}
exports.getStartOffsetOfAttribute = getStartOffsetOfAttribute;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZWxlbWVudHMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi8uLi8uLi8uLi8uLi8uLi9zcmMvY2RrL3NjaGVtYXRpY3MvbmctdXBkYXRlL2h0bWwtcGFyc2luZy9lbGVtZW50cy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUE7Ozs7OztHQU1HOzs7QUFFSCxtQ0FBeUQ7QUFFekQ7OztHQUdHO0FBQ0gsU0FBZ0IseUJBQXlCLENBQUMsSUFBWSxFQUFFLGFBQXFCO0lBQzNFLE1BQU0sUUFBUSxHQUFHLElBQUEsc0JBQWEsRUFBQyxJQUFJLEVBQUUsRUFBQyxzQkFBc0IsRUFBRSxJQUFJLEVBQUMsQ0FBQyxDQUFDO0lBQ3JFLE1BQU0sUUFBUSxHQUFjLEVBQUUsQ0FBQztJQUUvQixNQUFNLFVBQVUsR0FBRyxDQUFDLEtBQWtCLEVBQUUsRUFBRTtRQUN4QyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFO1lBQ2hCLE1BQU0sSUFBSSxHQUFHLENBQVksQ0FBQztZQUUxQixJQUFJLElBQUksQ0FBQyxVQUFVLEVBQUU7Z0JBQ25CLFVBQVUsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLENBQUM7YUFDN0I7WUFFRCxJQUFJLElBQUksQ0FBQyxLQUFLLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLElBQUksS0FBSyxhQUFhLENBQUMsV0FBVyxFQUFFLENBQUMsRUFBRTtnQkFDdkUsUUFBUSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQzthQUNyQjtRQUNILENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDO0lBRUYsVUFBVSxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUMsQ0FBQztJQUVoQyxPQUFPLFFBQVEsQ0FBQztBQUNsQixDQUFDO0FBckJELDhEQXFCQztBQUVEOzs7R0FHRztBQUNILFNBQWdCLDZCQUE2QixDQUFDLElBQVksRUFBRSxJQUFZLEVBQUUsUUFBa0I7SUFDMUYsT0FBTyx5QkFBeUIsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDO1NBQ3pDLE1BQU0sQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1NBQ3JELEdBQUcsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLE9BQU8sRUFBRSxJQUFJLENBQUMsQ0FBQyxDQUFDO0FBQzlELENBQUM7QUFKRCxzRUFJQztBQUVEOzs7R0FHRztBQUNILFNBQWdCLCtCQUErQixDQUFDLElBQVksRUFBRSxJQUFZLEVBQUUsS0FBZTtJQUN6RixPQUFPLHlCQUF5QixDQUFDLElBQUksRUFBRSxJQUFJLENBQUM7U0FDekMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLE9BQU8sRUFBRSxJQUFJLENBQUMsQ0FBQyxDQUFDO1NBQ3pFLEdBQUcsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLE9BQU8sRUFBRSxJQUFJLENBQUMsQ0FBQyxDQUFDO0FBQzlELENBQUM7QUFKRCwwRUFJQztBQUVELDRGQUE0RjtBQUM1RixTQUFTLG1CQUFtQixDQUFDLE9BQWdCLEVBQUUsYUFBcUI7SUFDbEUsT0FBTyxPQUFPLENBQUMsS0FBSyxJQUFJLE9BQU8sQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLElBQUksS0FBSyxhQUFhLENBQUMsV0FBVyxFQUFFLENBQUMsQ0FBQztBQUNoRyxDQUFDO0FBRUQsMEVBQTBFO0FBQzFFLFNBQWdCLHlCQUF5QixDQUFDLE9BQVksRUFBRSxhQUFxQjtJQUMzRSxPQUFPLE9BQU8sQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUMsV0FBVyxDQUFDO0FBQ25GLENBQUM7QUFGRCw4REFFQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogQGxpY2Vuc2VcbiAqIENvcHlyaWdodCBHb29nbGUgTExDIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogVXNlIG9mIHRoaXMgc291cmNlIGNvZGUgaXMgZ292ZXJuZWQgYnkgYW4gTUlULXN0eWxlIGxpY2Vuc2UgdGhhdCBjYW4gYmVcbiAqIGZvdW5kIGluIHRoZSBMSUNFTlNFIGZpbGUgYXQgaHR0cHM6Ly9hbmd1bGFyLmlvL2xpY2Vuc2VcbiAqL1xuXG5pbXBvcnQge0NoaWxkTm9kZSwgRWxlbWVudCwgcGFyc2VGcmFnbWVudH0gZnJvbSAncGFyc2U1JztcblxuLyoqXG4gKiBQYXJzZXMgYSBIVE1MIGZyYWdtZW50IGFuZCB0cmF2ZXJzZXMgYWxsIEFTVCBub2RlcyBpbiBvcmRlciBmaW5kIGVsZW1lbnRzIHRoYXRcbiAqIGluY2x1ZGUgdGhlIHNwZWNpZmllZCBhdHRyaWJ1dGUuXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBmaW5kRWxlbWVudHNXaXRoQXR0cmlidXRlKGh0bWw6IHN0cmluZywgYXR0cmlidXRlTmFtZTogc3RyaW5nKSB7XG4gIGNvbnN0IGRvY3VtZW50ID0gcGFyc2VGcmFnbWVudChodG1sLCB7c291cmNlQ29kZUxvY2F0aW9uSW5mbzogdHJ1ZX0pO1xuICBjb25zdCBlbGVtZW50czogRWxlbWVudFtdID0gW107XG5cbiAgY29uc3QgdmlzaXROb2RlcyA9IChub2RlczogQ2hpbGROb2RlW10pID0+IHtcbiAgICBub2Rlcy5mb3JFYWNoKG4gPT4ge1xuICAgICAgY29uc3Qgbm9kZSA9IG4gYXMgRWxlbWVudDtcblxuICAgICAgaWYgKG5vZGUuY2hpbGROb2Rlcykge1xuICAgICAgICB2aXNpdE5vZGVzKG5vZGUuY2hpbGROb2Rlcyk7XG4gICAgICB9XG5cbiAgICAgIGlmIChub2RlLmF0dHJzPy5zb21lKGF0dHIgPT4gYXR0ci5uYW1lID09PSBhdHRyaWJ1dGVOYW1lLnRvTG93ZXJDYXNlKCkpKSB7XG4gICAgICAgIGVsZW1lbnRzLnB1c2gobm9kZSk7XG4gICAgICB9XG4gICAgfSk7XG4gIH07XG5cbiAgdmlzaXROb2Rlcyhkb2N1bWVudC5jaGlsZE5vZGVzKTtcblxuICByZXR1cm4gZWxlbWVudHM7XG59XG5cbi8qKlxuICogRmluZHMgZWxlbWVudHMgd2l0aCBleHBsaWNpdCB0YWcgbmFtZXMgdGhhdCBhbHNvIGNvbnRhaW4gdGhlIHNwZWNpZmllZCBhdHRyaWJ1dGUuIFJldHVybnMgdGhlXG4gKiBhdHRyaWJ1dGUgc3RhcnQgb2Zmc2V0IGJhc2VkIG9uIHRoZSBzcGVjaWZpZWQgSFRNTC5cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGZpbmRBdHRyaWJ1dGVPbkVsZW1lbnRXaXRoVGFnKGh0bWw6IHN0cmluZywgbmFtZTogc3RyaW5nLCB0YWdOYW1lczogc3RyaW5nW10pIHtcbiAgcmV0dXJuIGZpbmRFbGVtZW50c1dpdGhBdHRyaWJ1dGUoaHRtbCwgbmFtZSlcbiAgICAuZmlsdGVyKGVsZW1lbnQgPT4gdGFnTmFtZXMuaW5jbHVkZXMoZWxlbWVudC50YWdOYW1lKSlcbiAgICAubWFwKGVsZW1lbnQgPT4gZ2V0U3RhcnRPZmZzZXRPZkF0dHJpYnV0ZShlbGVtZW50LCBuYW1lKSk7XG59XG5cbi8qKlxuICogRmluZHMgZWxlbWVudHMgdGhhdCBjb250YWluIHRoZSBnaXZlbiBhdHRyaWJ1dGUgYW5kIGNvbnRhaW4gYXQgbGVhc3Qgb25lIG9mIHRoZSBvdGhlclxuICogc3BlY2lmaWVkIGF0dHJpYnV0ZXMuIFJldHVybnMgdGhlIHByaW1hcnkgYXR0cmlidXRlJ3Mgc3RhcnQgb2Zmc2V0IGJhc2VkIG9uIHRoZSBzcGVjaWZpZWQgSFRNTC5cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGZpbmRBdHRyaWJ1dGVPbkVsZW1lbnRXaXRoQXR0cnMoaHRtbDogc3RyaW5nLCBuYW1lOiBzdHJpbmcsIGF0dHJzOiBzdHJpbmdbXSkge1xuICByZXR1cm4gZmluZEVsZW1lbnRzV2l0aEF0dHJpYnV0ZShodG1sLCBuYW1lKVxuICAgIC5maWx0ZXIoZWxlbWVudCA9PiBhdHRycy5zb21lKGF0dHIgPT4gaGFzRWxlbWVudEF0dHJpYnV0ZShlbGVtZW50LCBhdHRyKSkpXG4gICAgLm1hcChlbGVtZW50ID0+IGdldFN0YXJ0T2Zmc2V0T2ZBdHRyaWJ1dGUoZWxlbWVudCwgbmFtZSkpO1xufVxuXG4vKiogU2hvcnRoYW5kIGZ1bmN0aW9uIHRoYXQgY2hlY2tzIGlmIHRoZSBzcGVjaWZpZWQgZWxlbWVudCBjb250YWlucyB0aGUgZ2l2ZW4gYXR0cmlidXRlLiAqL1xuZnVuY3Rpb24gaGFzRWxlbWVudEF0dHJpYnV0ZShlbGVtZW50OiBFbGVtZW50LCBhdHRyaWJ1dGVOYW1lOiBzdHJpbmcpOiBib29sZWFuIHtcbiAgcmV0dXJuIGVsZW1lbnQuYXR0cnMgJiYgZWxlbWVudC5hdHRycy5zb21lKGF0dHIgPT4gYXR0ci5uYW1lID09PSBhdHRyaWJ1dGVOYW1lLnRvTG93ZXJDYXNlKCkpO1xufVxuXG4vKiogR2V0cyB0aGUgc3RhcnQgb2Zmc2V0IG9mIHRoZSBnaXZlbiBhdHRyaWJ1dGUgZnJvbSBhIFBhcnNlNSBlbGVtZW50LiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGdldFN0YXJ0T2Zmc2V0T2ZBdHRyaWJ1dGUoZWxlbWVudDogYW55LCBhdHRyaWJ1dGVOYW1lOiBzdHJpbmcpOiBudW1iZXIge1xuICByZXR1cm4gZWxlbWVudC5zb3VyY2VDb2RlTG9jYXRpb24uYXR0cnNbYXR0cmlidXRlTmFtZS50b0xvd2VyQ2FzZSgpXS5zdGFydE9mZnNldDtcbn1cbiJdfQ==