﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/amplify/AmplifyRequest.h>
#include <aws/amplify/Amplify_EXPORTS.h>
#include <aws/amplify/model/SourceUrlType.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Amplify {
namespace Model {

/**
 * <p>The request structure for the start a deployment request. </p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/StartDeploymentRequest">AWS
 * API Reference</a></p>
 */
class StartDeploymentRequest : public AmplifyRequest {
 public:
  AWS_AMPLIFY_API StartDeploymentRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "StartDeployment"; }

  AWS_AMPLIFY_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The unique ID for an Amplify app. </p>
   */
  inline const Aws::String& GetAppId() const { return m_appId; }
  inline bool AppIdHasBeenSet() const { return m_appIdHasBeenSet; }
  template <typename AppIdT = Aws::String>
  void SetAppId(AppIdT&& value) {
    m_appIdHasBeenSet = true;
    m_appId = std::forward<AppIdT>(value);
  }
  template <typename AppIdT = Aws::String>
  StartDeploymentRequest& WithAppId(AppIdT&& value) {
    SetAppId(std::forward<AppIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the branch to use for the deployment job. </p>
   */
  inline const Aws::String& GetBranchName() const { return m_branchName; }
  inline bool BranchNameHasBeenSet() const { return m_branchNameHasBeenSet; }
  template <typename BranchNameT = Aws::String>
  void SetBranchName(BranchNameT&& value) {
    m_branchNameHasBeenSet = true;
    m_branchName = std::forward<BranchNameT>(value);
  }
  template <typename BranchNameT = Aws::String>
  StartDeploymentRequest& WithBranchName(BranchNameT&& value) {
    SetBranchName(std::forward<BranchNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The job ID for this deployment that is generated by the
   * <code>CreateDeployment</code> request. </p>
   */
  inline const Aws::String& GetJobId() const { return m_jobId; }
  inline bool JobIdHasBeenSet() const { return m_jobIdHasBeenSet; }
  template <typename JobIdT = Aws::String>
  void SetJobId(JobIdT&& value) {
    m_jobIdHasBeenSet = true;
    m_jobId = std::forward<JobIdT>(value);
  }
  template <typename JobIdT = Aws::String>
  StartDeploymentRequest& WithJobId(JobIdT&& value) {
    SetJobId(std::forward<JobIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The source URL for the deployment that is used when calling
   * <code>StartDeployment</code> without <code>CreateDeployment</code>. The source
   * URL can be either an HTTP GET URL that is publicly accessible and downloads a
   * single .zip file, or an Amazon S3 bucket and prefix.</p>
   */
  inline const Aws::String& GetSourceUrl() const { return m_sourceUrl; }
  inline bool SourceUrlHasBeenSet() const { return m_sourceUrlHasBeenSet; }
  template <typename SourceUrlT = Aws::String>
  void SetSourceUrl(SourceUrlT&& value) {
    m_sourceUrlHasBeenSet = true;
    m_sourceUrl = std::forward<SourceUrlT>(value);
  }
  template <typename SourceUrlT = Aws::String>
  StartDeploymentRequest& WithSourceUrl(SourceUrlT&& value) {
    SetSourceUrl(std::forward<SourceUrlT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of source specified by the <code>sourceURL</code>. If the value is
   * <code>ZIP</code>, the source is a .zip file. If the value is
   * <code>BUCKET_PREFIX</code>, the source is an Amazon S3 bucket and prefix. If no
   * value is specified, the default is <code>ZIP</code>.</p>
   */
  inline SourceUrlType GetSourceUrlType() const { return m_sourceUrlType; }
  inline bool SourceUrlTypeHasBeenSet() const { return m_sourceUrlTypeHasBeenSet; }
  inline void SetSourceUrlType(SourceUrlType value) {
    m_sourceUrlTypeHasBeenSet = true;
    m_sourceUrlType = value;
  }
  inline StartDeploymentRequest& WithSourceUrlType(SourceUrlType value) {
    SetSourceUrlType(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_appId;
  bool m_appIdHasBeenSet = false;

  Aws::String m_branchName;
  bool m_branchNameHasBeenSet = false;

  Aws::String m_jobId;
  bool m_jobIdHasBeenSet = false;

  Aws::String m_sourceUrl;
  bool m_sourceUrlHasBeenSet = false;

  SourceUrlType m_sourceUrlType{SourceUrlType::NOT_SET};
  bool m_sourceUrlTypeHasBeenSet = false;
};

}  // namespace Model
}  // namespace Amplify
}  // namespace Aws
