﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/batch/Batch_EXPORTS.h>
#include <aws/batch/model/EksContainerEnvironmentVariable.h>
#include <aws/batch/model/EksContainerResourceRequirements.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace Batch {
namespace Model {

/**
 * <p>Object representing any Kubernetes overrides to a job definition that's used
 * in a <a
 * href="https://docs.aws.amazon.com/batch/latest/APIReference/API_SubmitJob.html">SubmitJob</a>
 * API operation.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/EksContainerOverride">AWS
 * API Reference</a></p>
 */
class EksContainerOverride {
 public:
  AWS_BATCH_API EksContainerOverride() = default;
  AWS_BATCH_API EksContainerOverride(Aws::Utils::Json::JsonView jsonValue);
  AWS_BATCH_API EksContainerOverride& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_BATCH_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>A pointer to the container that you want to override. The name must match a
   * unique container name that you wish to override.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  EksContainerOverride& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The override of the Docker image that's used to start the container.</p>
   */
  inline const Aws::String& GetImage() const { return m_image; }
  inline bool ImageHasBeenSet() const { return m_imageHasBeenSet; }
  template <typename ImageT = Aws::String>
  void SetImage(ImageT&& value) {
    m_imageHasBeenSet = true;
    m_image = std::forward<ImageT>(value);
  }
  template <typename ImageT = Aws::String>
  EksContainerOverride& WithImage(ImageT&& value) {
    SetImage(std::forward<ImageT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The command to send to the container that overrides the default command from
   * the Docker image or the job definition.</p>
   */
  inline const Aws::Vector<Aws::String>& GetCommand() const { return m_command; }
  inline bool CommandHasBeenSet() const { return m_commandHasBeenSet; }
  template <typename CommandT = Aws::Vector<Aws::String>>
  void SetCommand(CommandT&& value) {
    m_commandHasBeenSet = true;
    m_command = std::forward<CommandT>(value);
  }
  template <typename CommandT = Aws::Vector<Aws::String>>
  EksContainerOverride& WithCommand(CommandT&& value) {
    SetCommand(std::forward<CommandT>(value));
    return *this;
  }
  template <typename CommandT = Aws::String>
  EksContainerOverride& AddCommand(CommandT&& value) {
    m_commandHasBeenSet = true;
    m_command.emplace_back(std::forward<CommandT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The arguments to the entrypoint to send to the container that overrides the
   * default arguments from the Docker image or the job definition. For more
   * information, see <a
   * href="https://docs.docker.com/engine/reference/builder/#cmd">Dockerfile
   * reference: CMD</a> and <a
   * href="https://kubernetes.io/docs/tasks/inject-data-application/define-command-argument-container/">Define
   * a command an arguments for a pod</a> in the <i>Kubernetes documentation</i>.</p>
   */
  inline const Aws::Vector<Aws::String>& GetArgs() const { return m_args; }
  inline bool ArgsHasBeenSet() const { return m_argsHasBeenSet; }
  template <typename ArgsT = Aws::Vector<Aws::String>>
  void SetArgs(ArgsT&& value) {
    m_argsHasBeenSet = true;
    m_args = std::forward<ArgsT>(value);
  }
  template <typename ArgsT = Aws::Vector<Aws::String>>
  EksContainerOverride& WithArgs(ArgsT&& value) {
    SetArgs(std::forward<ArgsT>(value));
    return *this;
  }
  template <typename ArgsT = Aws::String>
  EksContainerOverride& AddArgs(ArgsT&& value) {
    m_argsHasBeenSet = true;
    m_args.emplace_back(std::forward<ArgsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The environment variables to send to the container. You can add new
   * environment variables, which are added to the container at launch. Or, you can
   * override the existing environment variables from the Docker image or the job
   * definition.</p>  <p>Environment variables cannot start with
   * "<code>AWS_BATCH</code>". This naming convention is reserved for variables that
   * Batch sets.</p>
   */
  inline const Aws::Vector<EksContainerEnvironmentVariable>& GetEnv() const { return m_env; }
  inline bool EnvHasBeenSet() const { return m_envHasBeenSet; }
  template <typename EnvT = Aws::Vector<EksContainerEnvironmentVariable>>
  void SetEnv(EnvT&& value) {
    m_envHasBeenSet = true;
    m_env = std::forward<EnvT>(value);
  }
  template <typename EnvT = Aws::Vector<EksContainerEnvironmentVariable>>
  EksContainerOverride& WithEnv(EnvT&& value) {
    SetEnv(std::forward<EnvT>(value));
    return *this;
  }
  template <typename EnvT = EksContainerEnvironmentVariable>
  EksContainerOverride& AddEnv(EnvT&& value) {
    m_envHasBeenSet = true;
    m_env.emplace_back(std::forward<EnvT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type and amount of resources to assign to a container. These override the
   * settings in the job definition. The supported resources include
   * <code>memory</code>, <code>cpu</code>, and <code>nvidia.com/gpu</code>. For more
   * information, see <a
   * href="https://kubernetes.io/docs/concepts/configuration/manage-resources-containers/">Resource
   * management for pods and containers</a> in the <i>Kubernetes
   * documentation</i>.</p>
   */
  inline const EksContainerResourceRequirements& GetResources() const { return m_resources; }
  inline bool ResourcesHasBeenSet() const { return m_resourcesHasBeenSet; }
  template <typename ResourcesT = EksContainerResourceRequirements>
  void SetResources(ResourcesT&& value) {
    m_resourcesHasBeenSet = true;
    m_resources = std::forward<ResourcesT>(value);
  }
  template <typename ResourcesT = EksContainerResourceRequirements>
  EksContainerOverride& WithResources(ResourcesT&& value) {
    SetResources(std::forward<ResourcesT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::String m_image;
  bool m_imageHasBeenSet = false;

  Aws::Vector<Aws::String> m_command;
  bool m_commandHasBeenSet = false;

  Aws::Vector<Aws::String> m_args;
  bool m_argsHasBeenSet = false;

  Aws::Vector<EksContainerEnvironmentVariable> m_env;
  bool m_envHasBeenSet = false;

  EksContainerResourceRequirements m_resources;
  bool m_resourcesHasBeenSet = false;
};

}  // namespace Model
}  // namespace Batch
}  // namespace Aws
