/***
  This file is part of avahi.

  avahi is free software; you can redistribute it and/or modify it
  under the terms of the GNU Lesser General Public License as
  published by the Free Software Foundation; either version 2.1 of the
  License, or (at your option) any later version.

  avahi is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
  or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General
  Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with avahi; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
  USA.
***/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>

#include <avahi-common/domain.h>
#include <avahi-common/malloc.h>
#include <avahi-common/error.h>
#ifdef HAVE_BONJOUR
#include <avahi-common/timeval.h>
#endif

#include "browse.h"
#include "log.h"

static void avahi_browse_domains_start(AvahiSDomainBrowser *b);

struct AvahiSDomainBrowser {
    int ref;

    AvahiServer *server;

    AvahiSRecordBrowser *record_browser;

    AvahiDomainBrowserType type;
    AvahiSDomainBrowserCallback callback;
    void* userdata;

    AvahiTimeEvent *defer_event;

    int all_for_now_scheduled;

#ifdef HAVE_BONJOUR
    AvahiIfIndex interface;
    AvahiTimeEvent *browse_error_event;
    AvahiTimeEvent *all_for_now_event;
    AvahiLookupFlags flags;
    AvahiWatch *watch;
    DNSServiceRef client;
#endif
    AVAHI_LLIST_FIELDS(AvahiSDomainBrowser, browser);
};

static void inc_ref(AvahiSDomainBrowser *b) {
    assert(b);
    assert(b->ref >= 1);

    b->ref++;
}

static void record_browser_callback(
    AvahiSRecordBrowser*rr,
    AvahiIfIndex interface,
    AvahiProtocol protocol,
    AvahiBrowserEvent event,
    AvahiRecord *record,
    AvahiLookupResultFlags flags,
    void* userdata) {

    AvahiSDomainBrowser *b = userdata;
    char *n = NULL;

    assert(rr);
    assert(b);

    if (event == AVAHI_BROWSER_ALL_FOR_NOW &&
        b->defer_event) {

        b->all_for_now_scheduled = 1;
        return;
    }

    /* Filter flags */
    flags &= AVAHI_LOOKUP_RESULT_CACHED | AVAHI_LOOKUP_RESULT_MULTICAST | AVAHI_LOOKUP_RESULT_WIDE_AREA;

    if (record) {
        assert(record->key->type == AVAHI_DNS_TYPE_PTR);
        n = record->data.ptr.name;

        if (b->type == AVAHI_DOMAIN_BROWSER_BROWSE) {
            AvahiStringList *l;

            /* Filter out entries defined statically */

            for (l = b->server->config.browse_domains; l; l = l->next)
                if (avahi_domain_equal((char*) l->text, n))
                    return;
        }

    }

    b->callback(b, interface, protocol, event, n, flags, b->userdata);
}

static void defer_callback(AvahiTimeEvent *e, void *userdata) {
    AvahiSDomainBrowser *b = userdata;
    AvahiStringList *l;

    assert(e);
    assert(b);

    assert(b->type == AVAHI_DOMAIN_BROWSER_BROWSE);

    avahi_time_event_free(b->defer_event);
    b->defer_event = NULL;

    /* Increase ref counter */
    inc_ref(b);

    for (l = b->server->config.browse_domains; l; l = l->next) {

        /* Check whether this object still exists outside our own
         * stack frame */
        if (b->ref <= 1)
            break;

        b->callback(b, AVAHI_IF_UNSPEC, AVAHI_PROTO_UNSPEC, AVAHI_BROWSER_NEW, (char*) l->text, AVAHI_LOOKUP_RESULT_STATIC, b->userdata);
    }

    if (b->ref > 1) {
        /* If the ALL_FOR_NOW event has already been scheduled, execute it now */

        if (b->all_for_now_scheduled)
            b->callback(b, AVAHI_IF_UNSPEC, AVAHI_PROTO_UNSPEC, AVAHI_BROWSER_ALL_FOR_NOW, NULL, 0, b->userdata);
    }

    /* Decrease ref counter */
    avahi_s_domain_browser_free(b);
}

AvahiSDomainBrowser *avahi_s_domain_browser_prepare(
    AvahiServer *server,
    AvahiIfIndex interface,
    AvahiProtocol protocol,
    const char *domain,
    AvahiDomainBrowserType type,
    AvahiLookupFlags flags,
    AvahiSDomainBrowserCallback callback,
    void* userdata) {

    static const char * const type_table[AVAHI_DOMAIN_BROWSER_MAX] = {
        "b",
        "db",
        "r",
        "dr",
        "lb"
    };

    AvahiSDomainBrowser *b;
    AvahiKey *k = NULL;
    char n[AVAHI_DOMAIN_NAME_MAX];
    int r;

    assert(server);
    assert(callback);

    AVAHI_CHECK_VALIDITY_RETURN_NULL(server, AVAHI_IF_VALID(interface), AVAHI_ERR_INVALID_INTERFACE);
    AVAHI_CHECK_VALIDITY_RETURN_NULL(server, AVAHI_PROTO_VALID(protocol), AVAHI_ERR_INVALID_PROTOCOL);
    AVAHI_CHECK_VALIDITY_RETURN_NULL(server, type < AVAHI_DOMAIN_BROWSER_MAX, AVAHI_ERR_INVALID_FLAGS);
    AVAHI_CHECK_VALIDITY_RETURN_NULL(server, !domain || avahi_is_valid_domain_name(domain), AVAHI_ERR_INVALID_DOMAIN_NAME);
    AVAHI_CHECK_VALIDITY_RETURN_NULL(server, AVAHI_FLAGS_VALID(flags, AVAHI_LOOKUP_USE_WIDE_AREA|AVAHI_LOOKUP_USE_MULTICAST), AVAHI_ERR_INVALID_FLAGS);

    if (!domain)
        domain = server->domain_name;

    if ((r = avahi_service_name_join(n, sizeof(n), type_table[type], "_dns-sd._udp", domain)) < 0) {
        avahi_server_set_errno(server, r);
        return NULL;
    }

    if (!(b = avahi_new(AvahiSDomainBrowser, 1))) {
        avahi_server_set_errno(server, AVAHI_ERR_NO_MEMORY);
        return NULL;
    }

    b->ref = 1;
    b->server = server;
    b->callback = callback;
    b->userdata = userdata;
    b->record_browser = NULL;
    b->type = type;
    b->all_for_now_scheduled = 0;
    b->defer_event = NULL;

    AVAHI_LLIST_PREPEND(AvahiSDomainBrowser, browser, server->domain_browsers, b);

#ifdef HAVE_BONJOUR
    b->interface = interface;
    b->client = NULL;
    b->watch = NULL;
    b->browse_error_event = NULL;
    b->all_for_now_event = NULL;
    b->flags = flags;
    avahi_browse_domains_start(b);
#else
    if (!(k = avahi_key_new(n, AVAHI_DNS_CLASS_IN, AVAHI_DNS_TYPE_PTR))) {
        avahi_server_set_errno(server, AVAHI_ERR_NO_MEMORY);
        goto fail;
    }

    if (!(b->record_browser = avahi_s_record_browser_prepare(server, interface, protocol, k, flags, record_browser_callback, b)))
        goto fail;

    avahi_key_unref(k);

    if (type == AVAHI_DOMAIN_BROWSER_BROWSE && b->server->config.browse_domains)
        b->defer_event = avahi_time_event_new(server->time_event_queue, NULL, defer_callback, b);
#endif

    return b;

fail:

    if (k)
        avahi_key_unref(k);

    avahi_s_domain_browser_free(b);

    return NULL;
}

void avahi_s_domain_browser_start(AvahiSDomainBrowser *b) {
    assert(b);

    if(b->record_browser)
        avahi_s_record_browser_start_query(b->record_browser);
}

void avahi_s_domain_browser_free(AvahiSDomainBrowser *b) {
    assert(b);

    assert(b->ref >= 1);
    if (--b->ref > 0)
        return;

    AVAHI_LLIST_REMOVE(AvahiSDomainBrowser, browser, b->server->domain_browsers, b);

#ifdef HAVE_BONJOUR
    if (b->browse_error_event) {
        avahi_time_event_free(b->browse_error_event);
        b->browse_error_event = NULL;
    }
    if (b->all_for_now_event) {
        avahi_time_event_free(b->all_for_now_event);
        b->all_for_now_event = NULL;
    }

    if (b->watch)
        b->server->poll_api->watch_free(b->watch);

    if (b->client)
        DNSServiceRefDeallocate(b->client);
#endif

    if (b->record_browser)
        avahi_s_record_browser_free(b->record_browser);

    if (b->defer_event)
        avahi_time_event_free(b->defer_event);

    avahi_free(b);
}

#ifdef HAVE_BONJOUR
static void enumerate_reply(DNSServiceRef service,
    DNSServiceFlags flags,
    uint32_t IfIndex,
    DNSServiceErrorType errorCode,
    const char *replyDomain,
    void *context) {
    AvahiSDomainBrowser *b = context;
    AvahiIfIndex interface;

    if (flags & kDNSServiceFlagsDefault)
        if (b->flags != AVAHI_DOMAIN_BROWSER_BROWSE_DEFAULT &&
            b->flags != AVAHI_DOMAIN_BROWSER_REGISTER_DEFAULT)
            return;

    interface = (IfIndex == kDNSServiceInterfaceIndexAny) ? AVAHI_IF_UNSPEC : IfIndex; 

    b->callback(b, interface, AVAHI_PROTO_UNSPEC, AVAHI_BROWSER_NEW, replyDomain, 0 /* flags */, b->userdata);
}

static void enumerate_error_callback(AvahiTimeEvent *e, void *userdata) {
    AvahiSDomainBrowser *b = userdata;

    if (b->browse_error_event) {
        avahi_time_event_free(b->browse_error_event);
        b->browse_error_event = NULL;
    }
    avahi_server_set_errno(b->server, AVAHI_ERR_FAILURE);
    b->callback(b,
        b->interface,
        AVAHI_PROTO_UNSPEC,
        AVAHI_BROWSER_FAILURE,
        NULL,
        0,
        b->userdata);
}

static void all_for_now_callback(AvahiTimeEvent *e, void* userdata) {
    AvahiSDomainBrowser *b = userdata;

    assert(e);
    assert(b);

    avahi_time_event_free(b->all_for_now_event);
    b->all_for_now_event = NULL;

    b->callback(b,
        AVAHI_IF_UNSPEC,
        AVAHI_PROTO_UNSPEC,
        AVAHI_BROWSER_ALL_FOR_NOW,
        NULL,
        0,
        b->userdata);
}

static void enumerate_socket_event(AvahiWatch *w, int fd, AvahiWatchEvent events, void *userdata) {
    AvahiSDomainBrowser *b = userdata;
    DNSServiceErrorType ret;

    assert(w);
    assert(fd >= 0);
    assert(events & AVAHI_WATCH_IN);

    assert (fd == DNSServiceRefSockFD(b->client));
    ret = DNSServiceProcessResult(b->client);
    if (ret != kDNSServiceErr_NoError) {
        if (b->watch) {
            b->server->poll_api->watch_free(b->watch);
            b->watch = NULL;
        }
        DNSServiceRefDeallocate(b->client);
        b->client = NULL;
        avahi_server_set_errno(b->server, AVAHI_ERR_DISCONNECTED);
        b->callback(b,
            b->interface,
            AVAHI_PROTO_UNSPEC,
            AVAHI_BROWSER_FAILURE,
            NULL,
            0,
            b->userdata);
    }
}

static void avahi_browse_domains_start(AvahiSDomainBrowser *b) {
    DNSServiceErrorType ret;
    DNSServiceFlags flags;
    struct timeval tv;

    if (b->flags == AVAHI_DOMAIN_BROWSER_BROWSE ||
        b->flags == AVAHI_DOMAIN_BROWSER_BROWSE_DEFAULT)
        flags = kDNSServiceFlagsBrowseDomains;
    else if (b->flags == AVAHI_DOMAIN_BROWSER_REGISTER ||
        b->flags == AVAHI_DOMAIN_BROWSER_REGISTER_DEFAULT)
        flags = kDNSServiceFlagsRegistrationDomains;
     
    ret =  DNSServiceEnumerateDomains(&b->client,
                                      flags,
                                      b->interface == AVAHI_IF_UNSPEC ?
                                          kDNSServiceInterfaceIndexAny :
                                          b->interface,
                                      enumerate_reply,
                                      b);
    if (ret != kDNSServiceErr_NoError || !b->client) {
        b->browse_error_event = avahi_time_event_new(b->server->time_event_queue,
NULL, enumerate_error_callback, b);
    } else {
        b->watch = b->server->poll_api->watch_new(b->server->poll_api, DNSServiceRefSockFD(b->client), AVAHI_WATCH_IN, enumerate_socket_event, b);

        /* Add a second */
        gettimeofday(&tv, NULL);
        avahi_timeval_add(&tv, 1000000);
        b->all_for_now_event = avahi_time_event_new(b->server->time_event_queue, &tv, all_for_now_callback, b);
    }
}
#endif

AvahiSDomainBrowser *avahi_s_domain_browser_new(
    AvahiServer *server,
    AvahiIfIndex interface,
    AvahiProtocol protocol,
    const char *domain,
    AvahiDomainBrowserType type,
    AvahiLookupFlags flags,
    AvahiSDomainBrowserCallback callback,
    void* userdata) {
        AvahiSDomainBrowser *b;

        b = avahi_s_domain_browser_prepare(server, interface, protocol, domain, type, flags, callback, userdata);
        avahi_s_domain_browser_start(b);

        return b;
}
